/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.expression.Identifier;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;

public class EnhancedForStatement extends AbstractStatement implements ForStatement {

  private final TypeName typeName;
  private final Identifier id;
  private final ExpressionCode expression;
  private final Statement statement;

  private EnhancedForStatement(TypeName typeName,
                               Identifier id,
                               ExpressionCode expression,
                               Statement statement) {
    this.typeName = typeName;
    this.id = id;
    this.expression = expression;
    this.statement = statement;
  }

  static Builder builder(Class<?> type, Identifier id, ExpressionCode expression) {
    checkNotNull(type, "type == null");
    checkNotNull(id, "id == null");
    checkNotNull(expression, "expression == null");
    ClassName className = ClassName.of(type);
    return new Builder(className, id, expression);
  }

  static Builder builder(TypeName typeName, Identifier id, ExpressionCode expression) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(id, "id == null");
    checkNotNull(expression, "expression == null");
    return new Builder(typeName, id, expression);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .writeWord("for")
        .writeWord('(').spaceOff()
        .writeTypeNameAsWord(typeName)
        .writeWord(id.name())
        .writeWord(':')
        .writeCodeElement(expression)
        .spaceOff().writeWord(')')
        .writeCodeElement(statement);
  }

  public static class Builder {

    private final TypeName typeName;
    private final Identifier id;
    private final ExpressionCode expression;

    private Builder(TypeName typeName, Identifier id, ExpressionCode expression) {
      this.typeName = typeName;
      this.id = id;
      this.expression = expression;
    }

    public final EnhancedForStatement block(BlockElement... elements) {
      Block block = Block.of(elements);
      return new EnhancedForStatement(typeName, id, expression, block);
    }

  }

}
