/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public class TryWithStatement extends TryElement implements TryStatement {

  private final ImmutableList<Resource> resources;
  private final Block block;

  private TryWithStatement(ImmutableList<Resource> resources,
                           Block block) {
    this.resources = resources;
    this.block = block;
  }

  static Builder builder(Resource... resources) {
    ImmutableList<Resource> list = ImmutableList.newListWithAll(resources);
    return new Builder(list);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("try").writeWord('(').spaceOff();

    Iterator<Resource> it = resources.iterator();
    if (it.hasNext()) {
      w.writeCodeElement(it.next());
      while (it.hasNext()) {
        w.spaceOff().writeWord(';');
        w.writeCodeElement(it.next());
      }
    }

    return w.spaceOff().writeWord(')').writeCodeElement(block);
  }

  @Override
  public final void acceptStatementOrBlockBuilder(StatementOrBlockBuilder builder) {
    builder.withStatement(this);
  }

  public static class Builder {

    private final ImmutableList<Resource> resources;

    private Builder(ImmutableList<Resource> resources) {
      this.resources = resources;
    }

    public final TryWithStatement block(Statement... statements) {
      Block block = Block.of(statements);
      return new TryWithStatement(resources, block);
    }

  }

}
