/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.type;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.MethodReference;
import br.com.objectos.code.java.expression.TypeWitness;
import br.com.objectos.code.java.expression.UnaryExpressionNotPlusMinus;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.JavaFileImportSet;

public abstract class ArrayTypeName implements ReferenceTypeName {

  public static ArrayTypeName of(ClassNameOrParameterizedTypeName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(PrimitiveTypeName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(TypeVariableName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(ArrayTypeName type) {
    checkNotNull(type, "type == null");
    return new MultiDimension(type);
  }

  static ArrayTypeName ofUnchecked(ClassNameOrParameterizedTypeName type) {
    return new SingleDimension(type);
  }

  static ArrayTypeName ofUnchecked(PrimitiveTypeName type) {
    return new SingleDimension(type);
  }

  static ArrayTypeName ofUnchecked(TypeVariableName type) {
    return new SingleDimension(type);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeTypeNameAsWord(this);
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    return componentTypeName().acceptJavaFileImportSet(set);
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitArrayTypeName(this, p);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    return componentTypeName().arrayCreationTypeName();
  }

  @Override
  public final CastExpression cast(UnaryExpressionNotPlusMinus expression) {
    return Expressions.cast(this, expression);
  }

  public abstract TypeName componentTypeName();

  public abstract TypeName deepComponentTypeName();

  public abstract int dimension();

  @Override
  public final MethodReference ref(String methodName) {
    return Expressions.ref(this, methodName);
  }

  @Override
  public final MethodReference ref(TypeWitness witness, String methodName) {
    return Expressions.ref(this, witness, methodName);
  }

  public final ArrayTypeName toArrayTypeName() {
    return new MultiDimension(this);
  }

  @Override
  public final String toString() {
    return componentTypeName() + "[]";
  }

  private static class SingleDimension extends ArrayTypeName {

    private final TypeName componentTypeName;

    SingleDimension(TypeName componentTypeName) {
      this.componentTypeName = componentTypeName;
    }

    @Override
    public final TypeName componentTypeName() {
      return componentTypeName;
    }

    @Override
    public final TypeName deepComponentTypeName() {
      return componentTypeName;
    }

    @Override
    public final int dimension() {
      return 1;
    }

  }

  private static class MultiDimension extends ArrayTypeName {

    private final ArrayTypeName componentTypeName;

    MultiDimension(ArrayTypeName componentTypeName) {
      this.componentTypeName = componentTypeName;
    }

    @Override
    public final TypeName componentTypeName() {
      return componentTypeName;
    }

    @Override
    public final TypeName deepComponentTypeName() {
      return componentTypeName.deepComponentTypeName();
    }

    @Override
    public final int dimension() {
      return componentTypeName.dimension() + 1;
    }

  }

}
