/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.declaration.AnnotationMember.AnnotatedMemberBuilder;
import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.LinkedHashMap;
import java.util.Map;

public class AnnotatedCode extends AbstractCodeElement {

  private final ClassName qualifiedName;
  private final ImmutableList<AnnotationMember> memberList;

  private AnnotatedCode(Builder builder) {
    qualifiedName = builder.qualifiedName;
    memberList = builder.memberList();
  }

  public static AnnotatedCode annotatedWith(Class<? extends Annotation> annotationType) {
    return builder(annotationType).build();
  }

  public static AnnotatedCode annotatedWith(ClassName className) {
    checkNotNull(className, "className == null");
    return builder(className).build();
  }

  public static Builder builder(Class<? extends Annotation> annotationType) {
    checkNotNull(annotationType, "annotationType == null");
    return new Builder(ClassName.ofUnchecked(annotationType));
  }

  public static Builder builder(ClassName className) {
    checkNotNull(className, "className == null");
    return new Builder(className);
  }

  public static AnnotatedCode of(Class<? extends Annotation> annotationType) {
    return builder(annotationType).build();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    switch (memberList.size()) {
      case 0:
        return acceptJavaWriter0(w);
      case 1:
        return acceptJavaWriter1(w);
      default:
        return acceptJavaWriter9(w);
    }
  }

  private CodeWriter acceptJavaWriter0(CodeWriter w) {
    return w.writeAnnotation(qualifiedName);
  }

  private CodeWriter acceptJavaWriter1(CodeWriter w) {
    return w.writeAnnotation(qualifiedName)
        .write('(')
        .spaceOff()
        .writeCodeElement(memberList.get(0))
        .write(')')
        .spaceOn();
  }

  private CodeWriter acceptJavaWriter9(CodeWriter w) {
    return w.writeAnnotation(qualifiedName);
  }

  public static class Builder {

    final ClassName qualifiedName;
    private final Map<String, AnnotatedMemberBuilder> memberMap = new LinkedHashMap<>();

    private Builder(ClassName qualifiedName) {
      this.qualifiedName = qualifiedName;
    }

    public final Builder addMember(String name, Class<?> type) {
      return addMember(name, ClassName.of(type));
    }

    public final Builder addMember(String name, ClassName qualifiedName) {
      member(name).addClass(qualifiedName);
      return this;
    }

    public final Builder addMember(String name, String value) {
      member(name).addString(value);
      return this;
    }

    public final AnnotatedCode build() {
      return new AnnotatedCode(this);
    }

    final ImmutableList<AnnotationMember> memberList() {
      GrowableList<AnnotationMember> memberList = GrowableList.newList();

      for (AnnotatedMemberBuilder builder : memberMap.values()) {
        AnnotationMember member = builder.build();
        memberList.add(member);
      }

      return memberList.toImmutableList();
    }

    private AnnotatedMemberBuilder member(String name) {
      checkNotNull(name, "name == null");
      return memberMap.computeIfAbsent(name, AnnotatedMemberBuilder::new);
    }

  }

}