/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.statement.Block;
import br.com.objectos.code.java.statement.BlockStatement;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.ImmutableSet;
import java.util.List;
import java.util.Set;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.VariableElement;

public final class ConstructorCode extends AbstractCodeElement implements ClassBodyElement {

  private static final ConstructorCode PRIVATE = _private().build();

  private final ImmutableSet<Modifier> modifierSet;
  private final ImmutableList<ParameterCode> parameters;
  private final Block block;

  ConstructorCode(Builder builder) {
    modifierSet = builder.modifierSet();
    parameters = builder.parameters();
    block = builder.block();
  }

  public static Builder _private() {
    return builder()._private();
  }

  public static Builder _protected() {
    return builder()._protected();
  }

  public static Builder _public() {
    return builder()._public();
  }

  public static Builder builder() {
    return new Builder();
  }

  public static ConstructorCode privateConstructor() {
    return PRIVATE;
  }

  public static Builder withSignature(ConstructorCodeElement element) {
    return builder().withSignature(element);
  }

  public static Builder withSignature(ExecutableElement element) {
    return builder().withSignature(element);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .writeModifierSet(modifierSet)
        .writeSimpleName()
        .writeParameters(parameters)
        .writeCodeElement(block);
  }

  @Override
  public final Kind kind() {
    return Kind.CONSTRUCTOR;
  }

  @Override
  public final String toString() {
    return acceptCodeWriter(CodeWriter
        .forToString()
        .pushSimpleName("Constructor"))
            .popSimpleName()
            .toString();
  }

  public static class Builder {

    private Modifier accessModifier;
    private final GrowableList<ParameterCode> parameters = GrowableList.newList();
    private final GrowableList<BlockStatement> body = GrowableList.newList();

    private Builder() {}

    public final ConstructorCode build() {
      return new ConstructorCode(this);
    }

    public final Builder _private() {
      return setAccessModifier(Modifier.PRIVATE);
    }

    public final Builder _protected() {
      return setAccessModifier(Modifier.PROTECTED);
    }

    public final Builder _public() {
      return setAccessModifier(Modifier.PUBLIC);
    }

    public final Builder body(BlockStatement... statements) {
      body.fluentAdd(statements);
      return this;
    }

    public final Builder body(Iterable<? extends BlockStatement> statements) {
      body.fluentAddAll(statements);
      return this;
    }

    public final Builder withParameter(Class<?> type, String name) {
      checkNotNull(type, "type == null");
      checkNotNull(name, "name == null");
      ClassName typeName = ClassName.ofUnchecked(type);
      ParameterCode parameter = ParameterCode.ofUnchecked(typeName, name);
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameter(ParameterCode parameter) {
      checkNotNull(parameter, "parameter == null");
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameter(TypeName typeName, String name) {
      checkNotNull(typeName, "typeName == null");
      checkNotNull(name, "name == null");
      ParameterCode parameter = ParameterCode.ofUnchecked(typeName, name);
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameters(Iterable<ParameterCode> parameters) {
      checkNotNull(parameters, "parameters == null");
      for (ParameterCode parameter : parameters) {
        withParameter(parameter);
      }
      return this;
    }

    public final Builder withParametersFrom(ConstructorCodeElement element) {
      for (ParameterCode parameter : element.parametersStream(ParameterCode::of)) {
        withParameter(parameter);
      }
      return this;
    }

    public final Builder withSignature(ConstructorCodeElement element) {
      checkNotNull(element, "element == null");
      return withAccessModifier(element).withParametersFrom(element);
    }

    public final Builder withSignature(ExecutableElement element) {
      checkNotNull(element, "element == null");
      return withAccessModifier(element).withParameterList(element);
    }

    final Block block() {
      return Block.of(body);
    }

    final ImmutableSet<Modifier> modifierSet() {
      return accessModifier != null
          ? ImmutableSet.newSetWith(accessModifier)
          : ImmutableSet.empty();
    }

    final ImmutableList<ParameterCode> parameters() {
      return parameters.toImmutableList();
    }

    private Builder setAccessModifier(Modifier modifier) {
      accessModifier = modifier;
      return this;
    }

    private Builder withAccessModifier(ConstructorCodeElement element) {
      if (element.isPublic()) {
        return _public();
      }

      if (element.isProtected()) {
        return _protected();
      }

      if (element.isPrivate()) {
        return _private();
      }

      return this;
    }

    private Builder withAccessModifier(ExecutableElement element) {
      Set<Modifier> modifiers = element.getModifiers();

      if (modifiers.contains(Modifier.PUBLIC)) {
        return _public();
      }

      if (modifiers.contains(Modifier.PROTECTED)) {
        return _protected();
      }

      if (modifiers.contains(Modifier.PRIVATE)) {
        return _private();
      }

      return this;
    }

    private Builder withParameterUnchecked(ParameterCode parameter) {
      parameters.add(parameter);
      return this;
    }

    private Builder withParameterList(ExecutableElement element) {
      List<? extends VariableElement> parameters = element.getParameters();
      for (VariableElement parameter : parameters) {
        withParameterUnchecked(ParameterCode.of(parameter));
      }
      return this;
    }

  }

}