/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.statement.VariableInitializer;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.NoTypeName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableSet;
import br.com.objectos.comuns.collections.ImmutableSet;
import javax.lang.model.element.Modifier;

public final class FieldCode extends AbstractCodeElement implements ClassBodyElement {

  private final ImmutableSet<Modifier> modifierSet;
  private final TypeName typeName;
  private final String name;
  private final VariableInitializer initializer;

  private FieldCode(Builder builder) {
    modifierSet = builder.modifierSet();
    typeName = builder.typeName;
    name = builder.name;
    initializer = builder.initializer;
  }

  public static Builder _final() {
    return builder()._final();
  }

  public static Builder _private() {
    return builder()._private();
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeModifierSet(modifierSet);
    w.writeTypeNameAsWord(typeName);
    w.writeWord(name);

    if (initializer != null) {
      w.writeWord('=');
      w.writePreSpace();
      w.spaceOff();
      w.writeCodeElement(initializer);
    }

    return w.write(';');
  }

  @Override
  public final Kind kind() {
    return Kind.FIELD;
  }

  public static class Builder {

    private final GrowableSet<Modifier> modifierSet = GrowableSet.newSet();
    private TypeName typeName = NoTypeName._void();
    private String name = "unnamed";
    private VariableInitializer initializer;

    private Builder() {}

    public final Builder _public() {
      return addModifier(Modifier.PUBLIC);
    }

    public final Builder _protected() {
      return addModifier(Modifier.PROTECTED);
    }

    public final Builder _private() {
      return addModifier(Modifier.PRIVATE);
    }

    public final Builder _static() {
      return addModifier(Modifier.DEFAULT);
    }

    public final Builder _final() {
      return addModifier(Modifier.FINAL);
    }

    public final Builder _synchronized() {
      return addModifier(Modifier.SYNCHRONIZED);
    }

    public final FieldCode build() {
      return new FieldCode(this);
    }

    public final Builder field(Class<?> type, String name) {
      return type(type).named(name);
    }

    public final Builder field(TypeName typeName, String name) {
      return type(typeName).named(name);
    }

    public final Builder named(String name) {
      this.name = checkNotNull(name, "name == null");
      return this;
    }

    public final Builder type(Class<?> type) {
      checkNotNull(type, "type == null");
      typeName = ClassName.ofUnchecked(type);
      return this;
    }

    public final Builder type(TypeName typeName) {
      this.typeName = checkNotNull(typeName, "typeName == null");
      return this;
    }

    public final Builder withInitializer(VariableInitializer initializer) {
      this.initializer = checkNotNull(initializer, "initializer == null");
      return this;
    }

    final ImmutableSet<Modifier> modifierSet() {
      return modifierSet.toImmutableSet();
    }

    private Builder addModifier(Modifier modifier) {
      modifierSet.add(modifier);
      return this;
    }

  }

}