/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.element;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public final class ImmutableCodeElement extends AbstractCodeElement {

  private final ImmutableList<CodeElement> elements;

  private ImmutableCodeElement(ImmutableList<CodeElement> elements) {
    this.elements = elements;
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeCodeElements(elements);
  }

  public final ImmutableCodeElement append(CodeElement element) {
    return new ImmutableCodeElement(append0(element));
  }

  private ImmutableList<CodeElement> append0(CodeElement element) {
    return GrowableList.<CodeElement> newList()
        .fluentAddAll(elements)
        .fluentAdd(element)
        .toImmutableList();
  }
  
  public static final class Builder {

    private final GrowableList<CodeElement> els = GrowableList.newList();

    private Builder() {}

    public final ImmutableCodeElement build() {
      return new ImmutableCodeElement(els.toImmutableList());
    }

    public final Builder with(String string) {
      checkNotNull(string, "string == null");
      return withCodeElement(CodeElements.ofRaw(string));
    }

    public final Builder withCloseAngle() {
      return withCodeElement(CodeElements.CLOSE_ANGLE);
    }

    public final Builder withCloseBrace() {
      return withCodeElement(CodeElements.CLOSE_BRACE);
    }

    public final Builder withCloseBracket() {
      return withCodeElement(CodeElements.CLOSE_BRACKET);
    }

    public final Builder withCloseParens() {
      return withCodeElement(CodeElements.CLOSE_PARENS);
    }

    public final Builder withCodeElement(CodeElement element) {
      els.addWithNullMessage(element, "element == null");
      return this;
    }

    public final Builder withCodeElements(Iterable<? extends CodeElement> elements) {
      els.addAll(elements);
      return this;
    }

    public final Builder withColon() {
      return withCodeElement(CodeElements.COLON);
    }

    public final Builder withCommaSeparated(CodeElement... elements) {
      return withCommaSeparated(ImmutableList.newListWithAll(elements));
    }

    public final Builder withCommaSeparated(Iterable<? extends CodeElement> elements) {
      checkNotNull(elements, "elements == null");

      Iterator<? extends CodeElement> it = elements.iterator();
      if (it.hasNext()) {
        els.add(it.next());
        while (it.hasNext()) {
          els.add(CodeElements.COMMA);
          els.add(it.next());
        }
      }

      return this;
    }

    public final Builder withDot() {
      return withCodeElement(CodeElements.DOT);
    }

    public final Builder withIndentIfNecessary() {
      return withCodeElement(CodeElements.INDENT_IF_NECESSARY);
    }

    public final Builder withOpenAngle() {
      return withCodeElement(CodeElements.OPEN_ANGLE);
    }

    public final Builder withOpenBrace() {
      return withCodeElement(CodeElements.OPEN_BRACE);
    }

    public final Builder withOpenBracket() {
      return withCodeElement(CodeElements.OPEN_BRACKET);
    }

    public final Builder withOpenParens() {
      return withCodeElement(CodeElements.OPEN_PARENS);
    }

    public final Builder withQuestionMark() {
      return withCodeElement(CodeElements.QUESTION_MARK);
    }

    public final Builder withSpaceOff() {
      return withCodeElement(CodeElements.SPACE_OFF);
    }

    public final Builder withSpaceOn() {
      return withCodeElement(CodeElements.SPACE_ON);
    }

    public final Builder withStringLiteral(String s) {
      checkNotNull(s, "s == null");
      return withCodeElement(CodeElements.ofStringLiteral(s));
    }

    public final Builder withTypeName(TypeName typeName) {
      return withCodeElement(CodeElements.ofTypeName(typeName));
    }

    public final Builder withWord(char c) {
      return withCodeElement(CodeElements.ofWord(c));
    }

    public final Builder withWord(String word) {
      checkNotNull(word, "word == null");
      return withCodeElement(CodeElements.ofWord(word));
    }

  }

}
