/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.declaration.BodyElement;
import br.com.objectos.code.java.element.NewLine;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.StreamIterable;

final class BodyElementFormatting extends Formatting {

  static final BodyElementFormatting CONSTRUCTORS
      = new BodyElementFormatting(BodyElement.Kind.CONSTRUCTOR);
  static final BodyElementFormatting FIELDS
      = new BodyElementFormatting(BodyElement.Kind.FIELD);
  static final BodyElementFormatting METHODS
      = new BodyElementFormatting(BodyElement.Kind.METHOD);
  static final BodyElementFormatting TYPES
      = new BodyElementFormatting(BodyElement.Kind.TYPE);

  private final BodyElement.Kind kind;
  private final PostAction postAction;

  private BodyElementFormatting(BodyElement.Kind kind) {
    this(kind, NewLinePostAction.INSTANCE);
  }

  private BodyElementFormatting(BodyElement.Kind kind,
                                PostAction postAction) {
    this.kind = kind;
    this.postAction = postAction;
  }

  @Override
  final FormattingAction newAction(FormattingAction nextAction) {
    return new ClassBodyElementFormattingAction(nextAction);
  }

  private class ClassBodyElementFormattingAction extends FormattingAction {

    private ClassBodyElementFormattingAction(FormattingAction nextAction) {
      super(nextAction);
    }

    @Override
    public final void consume(FormattingSource source) {
      while (source.hasElements()) {
        BodyElement element = source.getElement();
        consumeElement(element);
      }
      nextAction(source);
    }

    @Override
    public final void consumeElement(BodyElement element) {
      if (element.hasKind(kind)) {
        storeElement(element);
      } else {
        propagateElement(element);
      }
    }

    @Override
    final StreamIterable<BodyElement> elementsStream() {
      return postAction.applyTo(super.elementsStream());
    }

  }

  private static class NewLinePostAction extends PostAction {
    private static final NewLinePostAction INSTANCE = new NewLinePostAction();

    @Override
    final StreamIterable<BodyElement> applyTo(StreamIterable<BodyElement> elements) {
      GrowableList<BodyElement> result = GrowableList.newList();

      for (BodyElement element : elements) {
        result.add(element);
        result.add(NewLine.single());
      }

      return result;
    }
  }

  private abstract static class PostAction {
    abstract StreamIterable<BodyElement> applyTo(StreamIterable<BodyElement> elements);
  }

}