/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.declaration.BodyElement;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public abstract class BodyFormatter {

  private static final BodyFormatter DEFAULT = with(
      Formatting.newLine(),
      Formatting.fields(),
      Formatting.constructors(),
      Formatting.methods(),
      Formatting.types()
  );

  private static final BodyFormatter UNFORMATED = new UnformattedClassBodyFormatter();

  BodyFormatter() {}

  public static BodyFormatter defaultFormatter() {
    return DEFAULT;
  }

  public static BodyFormatter unformatted() {
    return UNFORMATED;
  }

  public static BodyFormatter with(Formatting... formattings) {
    return new StandardClassBodyFormatter(ImmutableList.newListWithAll(formattings));
  }

  public abstract <E extends BodyElement> ImmutableList<E> format(
      GrowableList<E> elements, Class<E> type);

  private static class StandardClassBodyFormatter extends BodyFormatter {

    private final ImmutableList<Formatting> formattings;

    private StandardClassBodyFormatter(ImmutableList<Formatting> formattings) {
      this.formattings = formattings;
    }

    @Override
    public final <E extends BodyElement> ImmutableList<E> format(
        GrowableList<E> elements, Class<E> type) {
      FormattingAction action = TailFormattingAction.getInstance();
      for (int i = formattings.size() - 1; i >= 0; i--) {
        Formatting formatting = formattings.get(i);
        action = formatting.newAction(action);
      }
      FormattingSource source = new ThisFormattingSource(elements);
      action.consume(source);
      return action.toImmutableList(type);
    }

    private static class ThisFormattingSource implements FormattingSource {
      private final Iterator<? extends BodyElement> elements;

      ThisFormattingSource(GrowableList<? extends BodyElement> elements) {
        this.elements = elements.iterator();
      }

      @Override
      public final BodyElement getElement() {
        return elements.next();
      }

      @Override
      public final boolean hasElements() {
        return elements.hasNext();
      }
    }

  }

  private static class UnformattedClassBodyFormatter extends BodyFormatter {
    @Override
    public final <E extends BodyElement> ImmutableList<E> format(
        GrowableList<E> elements, Class<E> type) {
      return elements.toImmutableList();
    }
  }

}