/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.expression.StatementExpression;
import br.com.objectos.code.java.statement.ForInit;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.ClassNameOrParameterizedTypeName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.code.java.type.TypeParameterName;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.StreamIterator;
import br.com.objectos.comuns.collections.StreamList;
import br.com.objectos.comuns.lang.Strings;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Iterator;
import java.util.Locale;
import javax.lang.model.element.Modifier;

public class CodeWriter {

  private static final int INDENTATION_SIZE = 2;

  private final StringBuilder out = new StringBuilder();

  private final ImportSet importSet;
  private final Deque<String> simpleNameStack = new ArrayDeque<>();

  private int length;
  private Indentation indentation = Indentation.start();
  private Space space = Space.OFF;

  private CodeWriter(ImportSet importSet) {
    this.importSet = importSet;
  }

  @Override
  public final String toString() {
    return out.toString();
  }

  public static CodeWriter forJavaFile(ImportSet importSet) {
    return new CodeWriter(importSet);
  }

  public static CodeWriter forToString() {
    ImportSet importSet = ImportSet.forToString();
    return new CodeWriter(importSet);
  }

  final CodeWriter beginBlock() {
    return writeWord('{').beginSection(Section.BLOCK);
  }

  public final CodeWriter beginSection(Section kind) {
    indentation = indentation.push(kind);
    return this;
  }

  final CodeWriter endBlock() {
    return endSection().writePreIndentation().write('}');
  }

  public final CodeWriter endSection() {
    indentation = indentation.pop();
    return this;
  }

  final CodeWriter join(String separator, Iterator<? extends CodeElement> elements) {
    Space cache = space;
    if (elements.hasNext()) {
      spaceOff().writeCodeElement(elements.next());
      while (elements.hasNext()) {
        write(separator).spaceOff().writeCodeElement(elements.next());
      }
    }
    space = cache;
    return this;
  }

  public final CodeWriter nextLine() {
    indentation.nextLine();
    return write('\n').resetLength().spaceOff();
  }

  public final CodeWriter popSimpleName() {
    simpleNameStack.pop();
    return this;
  }

  public final CodeWriter pushSimpleName(String simpleName) {
    simpleNameStack.push(simpleName);
    return this;
  }

  final CodeWriter resetLength() {
    length = 0;
    return this;
  }

  public final CodeWriter spaceOff() {
    space = Space.OFF;
    return this;
  }

  public final CodeWriter spaceOn() {
    space = Space.ON;
    return this;
  }

  public final String toJavaFile() {
    StringBuilder out = new StringBuilder();
    out.append(importSet);

    if (!importSet.isEmpty()) {
      out.append('\n');
      out.append('\n');
    }

    out.append(toString());
    return out.toString();
  }

  public final CodeWriter write(char c) {
    append(c);
    length++;
    return this;
  }

  public final CodeWriter write(String string) {
    append(string);
    length += string.length();
    return this;
  }

  public final CodeWriter writeAnnotation(ClassName qualifiedName) {
    return writeWord('@').writeTypeName(qualifiedName);
  }

  public final CodeWriter writeAnnotations(Iterable<? extends CodeElement> list) {
    for (CodeElement statement : list) {
      writeCodeElement(statement).nextLine();
    }
    return this;
  }

  public final CodeWriter writeBlock(Iterable<? extends CodeElement> list) {
    return beginBlock().block0(list).endBlock();
  }

  public final CodeWriter writeCast(TypeName type) {
    return writeWord('(').spaceOff()
        .writeTypeName(type)
        .spaceOff().writeWord(')');
  }

  public final CodeWriter writeCodeElement(CodeElement element) {
    return element.acceptCodeWriter(this);
  }

  public final CodeWriter writeCodeElements(Iterable<? extends CodeElement> elements) {
    for (CodeElement element : elements) {
      element.acceptCodeWriter(this);
    }
    return this;
  }

  public final CodeWriter writeControl(String control, ExpressionCode expression) {
    return writeWord(control).writeParenthesized(expression);
  }

  final CodeWriter writeIndentation(int count) {
    for (int i = 0; i < count * INDENTATION_SIZE; i++) {
      write(' ');
    }
    return this;
  }

  public final CodeWriter writeFor(ForInit init, ExpressionCode test, StatementExpression update) {
    writeWord("for");
    writeWord('(');

    spaceOff();
    writeCodeElement(init).spaceOff().writeWord(';');
    writeCodeElement(test).spaceOff().writeWord(';');
    writeCodeElement(update).spaceOff().writeWord(')');
    return this;
  }

  public final CodeWriter writeModifierSet(Iterable<Modifier> set) {
    for (Modifier modifier : set) {
      writeWord(modifier.name().toLowerCase(Locale.US));
    }
    return this;
  }

  public final CodeWriter writeImplementsIfNecessary(
      ImmutableList<ClassNameOrParameterizedTypeName> interfaces) {
    if (interfaces.isEmpty()) {
      return this;
    }

    writeWord("implements");

    StreamIterator<ClassNameOrParameterizedTypeName> iterator = interfaces.iterator();
    writeTypeNameAsWord(iterator.next());
    while (iterator.hasNext()) {
      write(", ");
      writeTypeName(iterator.next());
    }

    return this;
  }

  public final CodeWriter writeParameters(Iterable<? extends CodeElement> parameters) {
    return write('(').parameters0(parameters).write(')');
  }

  public final CodeWriter writeParenthesized(CodeElement element) {
    return writeWord('(').spaceOff()
        .writeCodeElement(element)
        .spaceOff().writeWord(')');
  }

  public final CodeWriter writePreIndentation() {
    if (length == 0) {
      writePreIndentation0();
    }
    return this;
  }

  public final CodeWriter writePreSpace() {
    space.writeTo(this);
    return this;
  }

  public final CodeWriter writeSimpleName() {
    return writeWord(peekSimpleName());
  }

  public final CodeWriter writeSimpleNameWith(
      StreamList<? extends TypeParameterName> typeParameters) {
    if (typeParameters.isEmpty()) {
      return writeSimpleName();
    } else {
      return simpleName0(typeParameters);
    }
  }

  public final CodeWriter writeStringLiteral(String string) {
    String escaped = Strings.escapeJava(string);
    return writeWord('"').write(escaped).write('"');
  }

  public final CodeWriter writeTypeName(TypeName typeName) {
    String s = importSet.get(typeName);
    return write(s);
  }

  public final CodeWriter writeTypeNameAsWord(TypeName typeName) {
    String s = importSet.get(typeName);
    return writeWord(s);
  }

  public final CodeWriter writeWord(char c) {
    return writePreWord().write(c).spaceOn();
  }

  public final CodeWriter writeWord(String word) {
    return writePreWord().write(word).spaceOn();
  }

  private void append(char c) {
    out.append(c);
  }

  private void append(String string) {
    out.append(string);
  }

  private CodeWriter block0(Iterable<? extends CodeElement> list) {
    Iterator<? extends CodeElement> iterator = list.iterator();
    if (iterator.hasNext()) {
      nextLine();
      writeCodeElement(iterator.next()).nextLine();
      while (iterator.hasNext()) {
        writeCodeElement(iterator.next()).nextLine();
      }
    }
    return this;
  }

  private CodeWriter parameters0(Iterable<? extends CodeElement> elements) {
    Iterator<? extends CodeElement> iterator = elements.iterator();

    Space cache = space;
    if (iterator.hasNext()) {
      spaceOff().writeCodeElement(iterator.next());
      while (iterator.hasNext()) {
        write(", ").spaceOff().writeCodeElement(iterator.next());
      }
    }
    space = cache;

    return this;
  }

  private String peekSimpleName() {
    if (simpleNameStack.isEmpty()) {
      throw new IllegalStateException("Stack of simple names is empty.");
    }
    return simpleNameStack.peek();
  }

  private void writePreIndentation0() {
    indentation.acceptCodeWriter(this);
  }

  private CodeWriter writePreWord() {
    return writePreIndentation().writePreSpace();
  }

  private CodeWriter simpleName0(StreamList<? extends TypeParameterName> list) {
    writePreWord().write(peekSimpleName()).write('<');

    Iterator<? extends TypeParameterName> iterator = list.iterator();
    if (iterator.hasNext()) {
      writeCodeElement(iterator.next());
      while (iterator.hasNext()) {
        write(", ").writeCodeElement(iterator.next());
      }
    }

    return write('>').spaceOn();
  }

}