/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.element.CodeElement;

abstract class Indentation implements CodeElement {

  Indentation() {}

  public static Indentation start() {
    return new Start();
  }

  public abstract Indentation nextLine();

  public abstract Indentation pop();

  public final Indentation push(Section section) {
    return new Rest(this, section);
  }

  abstract int indentation();

  private static class Start extends Indentation {
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w;
    }

    @Override
    public final Indentation nextLine() {
      return this;
    }

    @Override
    public final Indentation pop() {
      throw new IllegalStateException("Cannot pop an empty SectionStack.");
    }

    @Override
    final int indentation() {
      return 0;
    }
  }

  private static class Rest extends Indentation {

    private final Indentation parent;
    private final Section section;

    private boolean nextLine;

    Rest(Indentation parent, Section section) {
      this.parent = parent;
      this.section = section;
    }

    @Override
    public CodeWriter acceptCodeWriter(CodeWriter w) {
      return nextLine ? w.writeIndentation(indentation()) : parent.acceptCodeWriter(w);
    }

    @Override
    public final Indentation nextLine() {
      nextLine = true;
      return this;
    }

    @Override
    public final Indentation pop() {
      return parent;
    }

    @Override
    final int indentation() {
      return parent.indentation()
          + (nextLine ? section.indentationOnNextLine : section.indentation);
    }

  }

}