/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.declaration.PackageName;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableSet;
import br.com.objectos.comuns.collections.StreamIterator;

public class JavaFileImportSet extends ImportSet {

  private final GrowableSet<ClassName> qualifiedNameSet = GrowableSet.newSet();
  private final GrowableSet<String> simpleNameSet = GrowableSet.newSet();

  private final PackageName packageName;

  JavaFileImportSet(PackageName packageName) {
    this.packageName = packageName;
  }

  @Override
  public final boolean contains(ClassName qualifiedName) {
    return qualifiedNameSet.contains(qualifiedName);
  }

  @Override
  public final String get(TypeName typeName) {
    return typeName.acceptJavaFileImportSet(this);
  }

  @Override
  public final boolean isEmpty() {
    return size() == 0 && packageName.isUnnamed();
  }

  @Override
  public final int size() {
    return qualifiedNameSet.size();
  }

  @Override
  public final String toString() {
    return packageName.isUnnamed() ? toStringUnnamed() : toStringRegular();
  }

  public final boolean addQualifiedName(ClassName qualifiedName) {
    return qualifiedNameSet.add(qualifiedName);
  }

  public final boolean addSimpleName(String simpleName) {
    return simpleNameSet.add(simpleName);
  }

  public final boolean canSkipImport(PackageName otherPackageName) {
    return otherPackageName.is("java.lang") || packageName.equals(otherPackageName);
  }

  private String toStringRegular() {
    CodeWriter w = CodeWriter.forToString();

    w.writeWord("package");
    w.writeWord(packageName.toString());
    w.write(';');

    if (!qualifiedNameSet.isEmpty()) {
      w.nextLine();
    }

    for (ClassName className : qualifiedNameSet.sorted()) {
      w.nextLine();
      writeImport(w, className);
    }

    return w.toString();
  }

  private String toStringUnnamed() {
    CodeWriter w = CodeWriter.forToString();

    StreamIterator<ClassName> iterator = qualifiedNameSet.sorted().iterator();
    if (iterator.hasNext()) {
      writeImport(w, iterator.next());
      while (iterator.hasNext()) {
        w.nextLine();
        writeImport(w, iterator.next());
      }
    }

    return w.toString();
  }

  private void writeImport(CodeWriter w, ClassName className) {
    w.writeWord("import");
    w.writeWord(className.toString());
    w.write(';');
  }

}