/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;

public class DoStatement extends AbstractStatement {

  private final CodeElement body;
  private final ExpressionCode expression;

  private DoStatement(CodeElement body, ExpressionCode expression) {
    this.body = body;
    this.expression = expression;
  }

  public static DoStatement.Builder _do(BlockElement... elements) {
    return DoStatement.builder().blockBody().withBlockElement(elements);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("do");
    w.writeCodeElement(body);
    w.writeWord("while");
    w.writeWord('(');
    w.spaceOff();
    w.writeCodeElement(expression);
    w.spaceOff();
    w.writeWord(')');

    return w;
  }

  public static class Builder {

    private final StatementOrBlockBuilder statementOrBlock = new StatementOrBlockBuilder();

    public final DoStatement _while(ExpressionCode expression) {
      checkNotNull(expression, "expression == null");
      return new DoStatement(statementOrBlock.build(), expression);
    }

    public final Builder blockBody() {
      statementOrBlock.forceBlock();
      return this;
    }

    final Builder withBlockElement(BlockElement... elements) {
      checkNotNull(elements, "elements == null");
      for (BlockElement element : elements) {
        element.acceptStatementOrBlockBuilder(statementOrBlock);
      }
      return this;
    }

  }

}