/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.comuns.collections.ImmutableList;

public class SwitchStatement extends AbstractStatement {

  private final ExpressionCode expression;
  private final ImmutableList<SwitchElement> body;

  private SwitchStatement(ExpressionCode expression, ImmutableList<SwitchElement> body) {
    this.expression = expression;
    this.body = body;
  }

  public static SwitchStatement.Builder _switch(ExpressionCode expression) {
    return SwitchStatement.builder(expression);
  }

  static Builder builder(ExpressionCode expression) {
    checkNotNull(expression, "expression == null");
    return new Builder(expression);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("switch");
    w.writeWord('(').spaceOff();
    w.writeCodeElement(expression);
    w.spaceOff().writeWord(')');
    w.writeBlock(body);
    return w;
  }

  public static class Builder {

    private final ExpressionCode expression;

    private Builder(ExpressionCode expression) {
      this.expression = expression;
    }

    public final SwitchStatement block(SwitchElement... elements) {
      ImmutableList<SwitchElement> body = ImmutableList.newListWithAll(elements);
      return new SwitchStatement(expression, body);
    }

  }

}
