/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.expression.Identifier;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.StreamIterator;

public class TryCatchStatement extends AbstractStatement implements TryStatement {

  private final TryElement tryElement;
  private final ImmutableList<TypeName> exceptionTypes;
  private final Identifier id;
  private final Block block;

  private TryCatchStatement(TryElement tryElement,
                            ImmutableList<TypeName> exceptionTypes,
                            Identifier id,
                            Block block) {
    this.tryElement = tryElement;
    this.exceptionTypes = exceptionTypes;
    this.id = id;
    this.block = block;
  }

  static Builder builder(
      TryElement tryElement, ImmutableList<TypeName> exceptionTypes, Identifier id) {
    return new Builder(tryElement, exceptionTypes, id);
  }

  public final TryCatchFinallyStatement _finally(BlockStatement... statements) {
    Block block = Block.of(statements);
    return new TryCatchFinallyStatement(this, block);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeCodeElement(tryElement).spaceOn();
    w.writeWord("catch").writeWord('(').spaceOff();

    StreamIterator<TypeName> exIter = exceptionTypes.iterator();
    if (exIter.hasNext()) {
      w.writeTypeNameAsWord(exIter.next());
      while (exIter.hasNext()) {
        w.writeWord('|');
        w.writeTypeNameAsWord(exIter.next());
      }
    }

    w.writeCodeElement(id).spaceOff().writeWord(')');
    w.writeCodeElement(block);

    return w;
  }

  public static class Builder {

    private final TryElement tryElement;
    private final ImmutableList<TypeName> exceptionTypes;
    private final Identifier id;

    private Builder(TryElement tryElement,
                    ImmutableList<TypeName> exceptionTypes,
                    Identifier id) {
      this.tryElement = tryElement;
      this.exceptionTypes = exceptionTypes;
      this.id = id;
    }

    public final TryCatchStatement block(BlockElement... elements) {
      Block block = Block.of(elements);
      return new TryCatchStatement(tryElement, exceptionTypes, id, block);
    }

  }

}