/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;

public class DoStatement extends AbstractStatement {

  private final CodeElement body;
  private final ExpressionCode expression;

  private DoStatement(CodeElement body, ExpressionCode expression) {
    this.body = body;
    this.expression = expression;
  }

  public static DoStatement.Builder _do(BlockElement... elements) {
    return DoStatement.builder().blockBody().withBlockElement(elements);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("do");
    w.writeCodeElement(body);
    w.writeWord("while");
    w.writeWord('(');
    w.spaceOff();
    w.writeCodeElement(expression);
    w.spaceOff();
    w.writeWord(')');

    return w;
  }

  public static class Builder {

    private final StatementOrBlockBuilder statementOrBlock = new StatementOrBlockBuilder();

    public final DoStatement _while(ExpressionCode expression) {
      checkNotNull(expression, "expression == null");
      return new DoStatement(statementOrBlock.build(), expression);
    }

    public final Builder blockBody() {
      statementOrBlock.forceBlock();
      return this;
    }

    final Builder withBlockElement(BlockElement... elements) {
      checkNotNull(elements, "elements == null");
      for (BlockElement element : elements) {
        element.acceptStatementOrBlockBuilder(statementOrBlock);
      }
      return this;
    }

  }

}