/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import java.util.List;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.PrimitiveType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.WildcardType;
import javax.lang.model.util.SimpleTypeVisitor8;

class TypeNameFactory extends SimpleTypeVisitor8<TypeName, Void> {

  private static final TypeNameFactory INSTANCE = new TypeNameFactory();

  private TypeNameFactory() {}

  static TypeName ofUnchecked(TypeMirror type) {
    return type.accept(INSTANCE, null);
  }

  @Override
  public final TypeName visitDeclared(DeclaredType t, Void p) {
    TypeElement typeElement = (TypeElement) t.asElement();
    ClassNameOrParameterizedTypeName typeName = ClassName.of(typeElement);

    List<? extends TypeMirror> typeArgumentList = t.getTypeArguments();
    for (TypeMirror typeArgument : typeArgumentList) {
      typeName = typeName.withTypeArgument(typeArgument.accept(this, p));
    }

    return typeName;
  }

  @Override
  public final TypeName visitPrimitive(PrimitiveType t, Void p) {
    switch (t.getKind()) {
      case BOOLEAN:
        return PrimitiveTypeName._boolean();
      case BYTE:
        return PrimitiveTypeName._byte();
      case SHORT:
        return PrimitiveTypeName._short();
      case INT:
        return PrimitiveTypeName._int();
      case LONG:
        return PrimitiveTypeName._long();
      case CHAR:
        return PrimitiveTypeName._char();
      case FLOAT:
        return PrimitiveTypeName._float();
      case DOUBLE:
        return PrimitiveTypeName._double();
      default:
        throw new AssertionError();
    }
  }

  @Override
  public final TypeName visitWildcard(WildcardType t, Void p) {
    TypeMirror extendsBound = t.getExtendsBound();
    if (extendsBound != null) {
      return TypeNames.wildcardExtends(asRTN(extendsBound));
    }

    TypeMirror superBound = t.getSuperBound();
    if (superBound != null) {
      return TypeNames.wildcardSuper(asRTN(superBound));
    }

    return TypeNames.wildcard();
  }

  @Override
  protected final TypeName defaultAction(TypeMirror e, Void p) {
    throw new UnsupportedOperationException("Implement me");
  }
  
  private ReferenceTypeName asRTN(TypeMirror type) {
    TypeName typeName = type.accept(INSTANCE, null);
    return (ReferenceTypeName) typeName;
  }

}