/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.declaration.AnnotationMember.AnnotatedMemberBuilder;
import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.LinkedHashMap;
import java.util.Map;

public class AnnotatedCode extends AbstractCodeElement {

  private final ClassName qualifiedName;
  private final ImmutableList<AnnotationMember> memberList;

  private AnnotatedCode(Builder builder) {
    qualifiedName = builder.qualifiedName;
    memberList = builder.memberList();
  }

  public static AnnotatedCode annotatedWith(Class<? extends Annotation> annotationType) {
    return builder(annotationType).build();
  }

  public static AnnotatedCode annotatedWith(ClassName className) {
    checkNotNull(className, "className == null");
    return builder(className).build();
  }

  public static Builder builder(Class<? extends Annotation> annotationType) {
    checkNotNull(annotationType, "annotationType == null");
    return new Builder(ClassName.ofUnchecked(annotationType));
  }

  public static Builder builder(ClassName className) {
    checkNotNull(className, "className == null");
    return new Builder(className);
  }

  public static AnnotatedCode of(Class<? extends Annotation> annotationType) {
    return builder(annotationType).build();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    switch (memberList.size()) {
      case 0:
        return acceptJavaWriter0(w);
      case 1:
        return acceptJavaWriter1(w);
      default:
        return acceptJavaWriter9(w);
    }
  }

  private CodeWriter acceptJavaWriter0(CodeWriter w) {
    return w.writeAnnotation(qualifiedName);
  }

  private CodeWriter acceptJavaWriter1(CodeWriter w) {
    return w.writeAnnotation(qualifiedName)
        .write('(')
        .spaceOff()
        .writeCodeElement(memberList.get(0))
        .write(')')
        .spaceOn();
  }

  private CodeWriter acceptJavaWriter9(CodeWriter w) {
    return w.writeAnnotation(qualifiedName);
  }

  public static class Builder {

    final ClassName qualifiedName;
    private final Map<String, AnnotatedMemberBuilder> memberMap = new LinkedHashMap<>();

    private Builder(ClassName qualifiedName) {
      this.qualifiedName = qualifiedName;
    }

    public final Builder addMember(String name, Class<?> type) {
      return addMember(name, ClassName.of(type));
    }

    public final Builder addMember(String name, ClassName qualifiedName) {
      member(name).addClass(qualifiedName);
      return this;
    }

    public final Builder addMember(String name, String value) {
      member(name).addString(value);
      return this;
    }

    public final AnnotatedCode build() {
      return new AnnotatedCode(this);
    }

    final ImmutableList<AnnotationMember> memberList() {
      GrowableList<AnnotationMember> memberList = GrowableList.newList();

      for (AnnotatedMemberBuilder builder : memberMap.values()) {
        AnnotationMember member = builder.build();
        memberList.add(member);
      }

      return memberList.toImmutableList();
    }

    private AnnotatedMemberBuilder member(String name) {
      checkNotNull(name, "name == null");
      return memberMap.computeIfAbsent(name, AnnotatedMemberBuilder::new);
    }

  }

}