/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.BodyFormatter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeParameterName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;

public final class InterfaceCode extends TypeCode {

  private final InterfaceModifier modifier;
  private final String simpleName;
  private final ImmutableList<TypeParameterName> typeParameters = ImmutableList.empty();
  private final ImmutableList<InterfaceBodyElement> bodyElements;

  private InterfaceCode(Builder builder) {
    modifier = builder.modifier();
    simpleName = builder.simpleName;
    bodyElements = builder.bodyElements();
  }

  public static Builder _interface(String simpleName) {
    return builder()._interface(simpleName);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .pushSimpleName(simpleName)
        .writeCodeElement(modifier)
        .writeWord("interface")
        .writeSimpleNameWith(typeParameters)
        .writeBlock(bodyElements)
        .popSimpleName();
  }

  @Override
  public final String simpleName() {
    return simpleName;
  }

  public static final class Builder {

    private final InterfaceModifier.Builder modifier = InterfaceModifier.builder();
    private String simpleName = "Unnamed";
    private final GrowableList<InterfaceBodyElement> bodyElements = GrowableList.newList();

    private BodyFormatter formatter = BodyFormatter.defaultFormatter();

    private Builder() {}

    public final Builder _interface(String simpleName) {
      checkNotNull(simpleName, "simpleName == null");
      this.simpleName = simpleName;
      return this;
    }

    public final Builder body(InterfaceBodyElement... elements) {
      bodyElements.fluentAdd(elements);
      return this;
    }

    public final Builder body(Iterable<InterfaceBodyElement> elements) {
      return withBodyElements(elements);
    }

    public final InterfaceCode build() {
      return new InterfaceCode(this);
    }

    public final InterfaceCode buildWith(BodyFormatter formatter) {
      this.formatter = checkNotNull(formatter, "formatter == null");
      return build();
    }

    public final Builder withBodyElements(Iterable<InterfaceBodyElement> elements) {
      bodyElements.addAll(elements);
      return this;
    }

    public final Builder withModifier(InterfaceModifier modifier) {
      checkNotNull(modifier, "modifier == null");
      this.modifier.withModifier(modifier);
      return this;
    }

    final ImmutableList<InterfaceBodyElement> bodyElements() {
      return formatter.format(bodyElements, InterfaceBodyElement.class);
    }

    final InterfaceModifier modifier() {
      return modifier.build();
    }

  }

}