/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.type.EnclosingElement;
import javax.lang.model.element.PackageElement;

public class PackageName implements EnclosingElement {

  private static final PackageName UNNAMED = new PackageName("");

  private final String packageName;

  private PackageName(String packageName) {
    this.packageName = packageName;
  }

  public static PackageName named(String name) {
    checkNotNull(name, "name == null");
    return new PackageName(name);
  }

  public static PackageName of(Class<?> type) {
    checkNotNull(type, "type == null");
    return of(type.getPackage());
  }

  public static PackageName of(Package pkg) {
    checkNotNull(pkg, "pkg == null");
    return new PackageName(pkg.getName());
  }

  public static PackageName of(PackageElement element) {
    checkNotNull(element, "element == null");
    return new PackageName(element.getQualifiedName().toString());
  }

  public static PackageName unnamed() {
    return UNNAMED;
  }

  @Override
  public final boolean equals(Object obj) {
    if (!(obj instanceof PackageName)) {
      return false;
    }
    PackageName that = (PackageName) obj;
    return packageName.equals(that.packageName);
  }

  @Override
  public final int hashCode() {
    return packageName.hashCode();
  }

  public final boolean is(String name) {
    return packageName.equals(name);
  }

  public final boolean isUnnamed() {
    return is("");
  }

  @Override
  public final PackageName packageName() {
    return this;
  }

  @Override
  public final String toString() {
    return packageName;
  }

}