/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.expression;

import br.com.objectos.code.java.element.AbstractDefaultCodeElement;
import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.element.ImmutableCodeElement;
import br.com.objectos.comuns.collections.StreamIterable;
import java.util.Arrays;
import java.util.function.Function;

final class ArrayAccessImpl extends AbstractDefaultCodeElement
    implements ArrayAccess {

  private static final Function<ExpressionCode, CodeElement> TO_ACCESS0
      = new Function<ExpressionCode, CodeElement>() {
        @Override
        public final CodeElement apply(ExpressionCode t) {
          return access0(t);
        }
      };

  private ArrayAccessImpl(ImmutableCodeElement delegate) {
    super(delegate);
  }

  static ArrayAccess arrayAccess0(ArrayReferenceExpression ref,
      ExpressionCode... expressions) {
    return arrayAccess0(ref, Arrays.asList(expressions));
  }

  static ArrayAccess arrayAccess0(
      ArrayReferenceExpression ref, Iterable<? extends ExpressionCode> expressions) {
    return new ArrayAccessImpl(
        ImmutableCodeElement.builder()
            .withCodeElement(ref)
            .withCodeElements(
                StreamIterable.adapt(expressions).map(TO_ACCESS0)
            )
            .build()
    );
  }

  private static CodeElement access0(ExpressionCode expression) {
    return ImmutableCodeElement.builder()
        .withSpaceOff()
        .withOpenBracket()
        .withSpaceOff()
        .withCodeElement(expression)
        .withSpaceOff()
        .withCloseBracket()
        .build();
  }

  @Override
  public final FieldAccess id(Identifier id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  public final FieldAccess id(String id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  protected final ArrayReferenceExpression selfArrayReferenceExpression() {
    return this;
  }

  @Override
  protected final Callee selfCallee() {
    return this;
  }

  @Override
  protected final ConditionalAndExpression selfConditionalAndExpression() {
    return this;
  }

  @Override
  protected final LeftHandSide selfLeftHandSide() {
    return this;
  }

  @Override
  protected final MultiplicativeExpression selfMultiplicativeExpression() {
    return this;
  }

  @Override
  protected final MethodReferenceReferenceExpression selfMethodReferenceReferenceExpression() {
    return this;
  }

  @Override
  protected final PostfixExpression selfPostfixExpression() {
    return this;
  }

  @Override
  protected final RelationalExpression selfRelationalExpression() {
    return this;
  }

}
