/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.MethodReference;
import br.com.objectos.code.java.expression.TypeWitness;
import br.com.objectos.code.java.expression.UnaryExpressionNotPlusMinus;
import br.com.objectos.code.java.io.JavaFileImportSet;

public abstract class ArrayTypeName extends AbstractTypeName implements ReferenceTypeName {

  ArrayTypeName() {}

  public static ArrayTypeName of(ArrayTypeName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(ClassNameOrParameterizedTypeName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(PrimitiveTypeName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ArrayTypeName of(TypeVariableName type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  static ArrayTypeName ofUnchecked(ArrayTypeName type) {
    return new MultiDimension(type);
  }

  static ArrayTypeName ofUnchecked(ClassNameOrParameterizedTypeName type) {
    return new SingleDimension(type);
  }

  static ArrayTypeName ofUnchecked(PrimitiveTypeName type) {
    return new SingleDimension(type);
  }

  static ArrayTypeName ofUnchecked(TypeVariableName type) {
    return new SingleDimension(type);
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    return componentTypeName().acceptJavaFileImportSet(set) + "[]";
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitArrayTypeName(this, p);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    return componentTypeName().arrayCreationTypeName();
  }

  @Override
  public final CastExpression cast(UnaryExpressionNotPlusMinus expression) {
    return Expressions.cast(this, expression);
  }

  public abstract TypeName componentTypeName();

  public abstract TypeName deepComponentTypeName();

  public abstract int dimension();

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof ArrayTypeName)) {
      return false;
    }
    return toString().equals(obj.toString());
  }

  @Override
  public final int hashCode() {
    return toString().hashCode();
  }

  public abstract String printVarArgsSymbol();

  @Override
  public final MethodReference ref(String methodName) {
    return Expressions.ref(this, methodName);
  }

  @Override
  public final MethodReference ref(TypeWitness witness, String methodName) {
    return Expressions.ref(this, witness, methodName);
  }

  @Override
  public final ArrayTypeName toArrayTypeName() {
    return new MultiDimension(this);
  }

  @Override
  public final String toString() {
    return componentTypeName() + "[]";
  }

  private static class SingleDimension extends ArrayTypeName {

    private final TypeName componentTypeName;

    SingleDimension(TypeName componentTypeName) {
      this.componentTypeName = componentTypeName;
    }

    @Override
    public final TypeName componentTypeName() {
      return componentTypeName;
    }

    @Override
    public final TypeName deepComponentTypeName() {
      return componentTypeName;
    }

    @Override
    public final int dimension() {
      return 1;
    }

    @Override
    public final String printVarArgsSymbol() {
      return "...";
    }

  }

  private static class MultiDimension extends ArrayTypeName {

    private final ArrayTypeName componentTypeName;

    MultiDimension(ArrayTypeName componentTypeName) {
      this.componentTypeName = componentTypeName;
    }

    @Override
    public final TypeName componentTypeName() {
      return componentTypeName;
    }

    @Override
    public final TypeName deepComponentTypeName() {
      return componentTypeName.deepComponentTypeName();
    }

    @Override
    public final int dimension() {
      return componentTypeName.dimension() + 1;
    }

    @Override
    public final String printVarArgsSymbol() {
      return "[]" + componentTypeName.printVarArgsSymbol();
    }

  }

}
