/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.UnaryExpression;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.JavaFileImportSet;

public enum PrimitiveTypeName implements TypeName {

  BOOLEAN("boolean", Boolean.class),
  BYTE("byte", Byte.class),
  SHORT("short", Short.class),
  INT("int", Integer.class),
  LONG("long", Long.class),
  CHAR("char", Character.class),
  FLOAT("float", Float.class),
  DOUBLE("double", Double.class);

  private final String name;
  private final Class<?> wrapperClass;

  private PrimitiveTypeName(String name, Class<?> wrapperClass) {
    this.name = name;
    this.wrapperClass = wrapperClass;
  }

  public static PrimitiveTypeName _boolean() {
    return PrimitiveTypeName.BOOLEAN;
  }

  public static PrimitiveTypeName _byte() {
    return PrimitiveTypeName.BYTE;
  }

  public static PrimitiveTypeName _short() {
    return PrimitiveTypeName.SHORT;
  }

  public static PrimitiveTypeName _int() {
    return PrimitiveTypeName.INT;
  }

  public static PrimitiveTypeName _long() {
    return PrimitiveTypeName.LONG;
  }

  public static PrimitiveTypeName _char() {
    return PrimitiveTypeName.CHAR;
  }

  public static PrimitiveTypeName _float() {
    return PrimitiveTypeName.FLOAT;
  }

  public static PrimitiveTypeName _double() {
    return PrimitiveTypeName.DOUBLE;
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeTypeNameAsWord(this);
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    return name;
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitPrimitiveType(this, p);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    return this;
  }

  public final CastExpression cast(UnaryExpression expression) {
    return Expressions.cast(this, expression);
  }

  @Override
  public final boolean isVarArgs() {
    return false;
  }

  public final ArrayTypeName toArrayTypeName() {
    return ArrayTypeName.ofUnchecked(this);
  }

  @Override
  public final String toString() {
    return name;
  }

  public final ClassName wrapperClassName() {
    return ClassName.of(wrapperClass);
  }

}