/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.statement.Block;
import br.com.objectos.code.java.statement.BlockStatement;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.NoTypeName;
import br.com.objectos.code.java.type.PrimitiveTypeName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.Set;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;

public final class MethodCode extends AbstractCodeElement
    implements ClassBodyElement, InterfaceBodyElement {

  private final MethodModifier modifier;
  private final TypeName typeName;
  private final String name;

  private final ImmutableList<AnnotatedCode> annotations;
  private final ImmutableList<ParameterCode> parameters;
  private final Block block;

  private MethodCode(Builder builder) {
    modifier = builder.modifier();
    typeName = builder.typeName;
    name = builder.name;
    annotations = builder.annotations();
    parameters = builder.parameters();
    block = builder.block();
  }

  public static Builder _public() {
    return builder()._public();
  }

  public static Builder _protected() {
    return builder()._protected();
  }

  public static Builder _private() {
    return builder()._private();
  }

  public static Builder _abstract() {
    return builder()._abstract();
  }

  public static Builder _default() {
    return builder()._default();
  }

  public static Builder _static() {
    return builder()._static();
  }

  public static Builder _final() {
    return builder()._final();
  }

  public static Builder _synchronized() {
    return builder()._synchronized();
  }

  public static Builder _native() {
    return builder()._native();
  }

  public static Builder _strictfp() {
    return builder()._strictfp();
  }

  public static Builder _int() {
    return builder()._int();
  }

  public static Builder _void() {
    return builder()._void();
  }

  public static Builder annotatedWith(Class<? extends Annotation> annotationType) {
    return builder().annotatedWith(annotationType);
  }

  public static Builder builder() {
    return new Builder();
  }

  public static Builder returning(Class<?> returnType) {
    return builder().returning(returnType);
  }

  public static Builder returning(TypeName returnType) {
    return builder().returning(returnType);
  }

  public static Builder withSameAccessLevel(ExecutableElement element) {
    return new Builder().withSameAccessLevel(element);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    if (!block.isEmpty()) {
      return acceptJavaWriter0(w).writeCodeElement(block);
    } else {
      return acceptJavaWriter0(w).write(';');
    }
  }

  @Override
  public final Kind kind() {
    return Kind.METHOD;
  }

  private CodeWriter acceptJavaWriter0(CodeWriter w) {
    return w
        .writeAnnotations(annotations)
        .writeCodeElement(modifier)
        .writeTypeNameAsWord(typeName)
        .writeWord(name)
        .writeParameters(parameters);
  }

  public static class Builder {

    private final GrowableList<AnnotatedCode> annotations = GrowableList.newList();
    private final MethodModifier.Builder modifier = MethodModifier.builder();
    private TypeName typeName = NoTypeName._void();
    private String name = "unnamed";
    private final GrowableList<ParameterCode> parameters = GrowableList.newList();
    private final GrowableList<BlockStatement> block = GrowableList.newList();

    private Builder() {}

    public final Builder _public() {
      modifier._public();
      return this;
    }

    public final Builder _protected() {
      modifier._protected();
      return this;
    }

    public final Builder _private() {
      modifier._private();
      return this;
    }

    public final Builder _abstract() {
      modifier._abstract();
      return this;
    }

    public final Builder _default() {
      modifier._default();
      return this;
    }

    public final Builder _static() {
      modifier._static();
      return this;
    }

    public final Builder _final() {
      modifier._final();
      return this;
    }

    public final Builder _synchronized() {
      modifier._synchronized();
      return this;
    }

    public final Builder _native() {
      modifier._native();
      return this;
    }

    public final Builder _strictfp() {
      modifier._strictfp();
      return this;
    }

    public final Builder _int() {
      return setTypeNameUnchecked(PrimitiveTypeName._int());
    }

    public final Builder _void() {
      return returning(NoTypeName._void());
    }

    public final Builder annotatedWith(AnnotatedCode annotation) {
      annotations.add(annotation);
      return this;
    }

    public final Builder annotatedWith(Class<? extends Annotation> annotationType) {
      return annotatedWith(AnnotatedCode.annotatedWith(annotationType));
    }

    public final Builder body(BlockStatement... statements) {
      block.fluentAdd(statements);
      return this;
    }

    public final Builder body(Iterable<? extends BlockStatement> statements) {
      block.addAll(statements);
      return this;
    }

    public final MethodCode build() {
      return new MethodCode(this);
    }

    public final Builder named(String name) {
      this.name = checkNotNull(name, "name == null");
      return this;
    }

    public final Builder returning(Class<?> returnType) {
      checkNotNull(returnType, "returnType == null");
      return setTypeNameUnchecked(ClassName.ofUnchecked(returnType));
    }

    public final Builder returning(TypeName typeName) {
      checkNotNull(typeName, "typeName == null");
      return setTypeNameUnchecked(typeName);
    }

    public final Builder withAccessLevel(AccessLevel accessLevel) {
      checkNotNull(accessLevel, "accessLevel == null");
      switch (accessLevel) {
        case PUBLIC:
          return _public();
        case PROTECTED:
          return _protected();
        case PRIVATE:
          return _private();
        default:
          return this;
      }
    }

    public final Builder withAnnotations(Iterable<AnnotatedCode> annotations) {
      this.annotations.addAll(annotations);
      return this;
    }

    public final Builder withModifier(MethodModifier modifier) {
      this.modifier.withModifier(modifier);
      return this;
    }

    public final Builder withParameter(Class<?> type, String name) {
      checkNotNull(type, "type == null");
      checkNotNull(name, "name == null");
      ClassName typeName = ClassName.ofUnchecked(type);
      ParameterCode parameter = ParameterCode.ofUnchecked(typeName, name);
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameter(ParameterCode parameter) {
      checkNotNull(parameter, "parameter == null");
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameter(ParameterTypeName typeName, String name) {
      checkNotNull(typeName, "typeName == null");
      checkNotNull(name, "name == null");
      ParameterCode parameter = ParameterCode.ofUnchecked(typeName, name);
      return withParameterUnchecked(parameter);
    }

    public final Builder withParameters(Iterable<ParameterCode> parameters) {
      checkNotNull(parameters, "parameters == null");
      for (ParameterCode parameter : parameters) {
        withParameter(parameter);
      }
      return this;
    }

    public final Builder withSameAccessLevel(ExecutableElement element) {
      checkNotNull(element, "element == null");
      Set<Modifier> modifierSet = element.getModifiers();

      if (modifierSet.contains(Modifier.PUBLIC)) {
        return _public();
      } else if (modifierSet.contains(Modifier.PROTECTED)) {
        return _protected();
      } else if (modifierSet.contains(Modifier.PRIVATE)) {
        return _private();
      } else {
        return this;
      }
    }

    final ImmutableList<AnnotatedCode> annotations() {
      return annotations.toImmutableList();
    }

    final Block block() {
      return Block.of(block);
    }

    final MethodModifier modifier() {
      return modifier.build();
    }

    final ImmutableList<ParameterCode> parameters() {
      return parameters.toImmutableList();
    }

    final String name() {
      return name;
    }

    private Builder setTypeNameUnchecked(TypeName typeName) {
      this.typeName = typeName;
      return this;
    }

    private Builder withParameterUnchecked(ParameterCode parameter) {
      parameters.add(parameter);
      return this;
    }

  }

}
