/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.ImportSet;
import br.com.objectos.code.java.io.JavaFile;
import br.com.objectos.code.java.type.ClassName;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import javax.annotation.processing.Filer;
import javax.tools.JavaFileObject;

abstract class TypeCode extends AbstractCodeElement implements ClassBodyElement {

  TypeCode() {}

  @Override
  public final Kind kind() {
    return Kind.TYPE;
  }

  public abstract String simpleName();

  public final JavaFile toJavaFile(PackageName packageName) {
    checkNotNull(packageName, "packageName == null");
    return new TypeCodeJavaFile(packageName, this);
  }

  private static class TypeCodeJavaFile implements JavaFile {

    private final PackageName packageName;
    private final TypeCode typeCode;

    TypeCodeJavaFile(PackageName packageName, TypeCode typeCode) {
      this.packageName = packageName;
      this.typeCode = typeCode;
    }

    @Override
    public final String toString() {
      ImportSet importSet = ImportSet.forPackageName(packageName);
      CodeWriter writer = CodeWriter.forJavaFile(importSet);
      typeCode.acceptCodeWriter(writer);
      return writer.toJavaFile();
    }

    @Override
    public final void writeTo(Filer filer) {
      try {
        writeTo0(filer);
      } catch (IOException e) {
        throw new UncheckedIOException(e);
      }
    }

    private void writeTo0(Filer filer) throws IOException {
      ClassName className = packageName.nestedClass(typeCode.simpleName());
      JavaFileObject object = filer.createSourceFile(className.toString());
      try (Writer writer = object.openWriter()) {
        writer.write(toString());
      }
    }

  }

}