/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.expression;

import br.com.objectos.code.java.element.AbstractDefaultCodeElement;
import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.element.ImmutableCodeElement;
import br.com.objectos.code.java.type.ArrayTypeName;
import br.com.objectos.comuns.collections.GrowableList;
import java.util.Arrays;
import java.util.Iterator;

final class ArrayCreationExpressionImpl
    extends AbstractDefaultCodeElement
    implements ArrayCreationExpression {

  private ArrayCreationExpressionImpl(ImmutableCodeElement delegate) {
    super(delegate);
  }

  static ArrayCreationExpression _new0(ArrayTypeName type, ExpressionCode... dims) {
    return _new0(type, Arrays.asList(dims));
  }

  static ArrayCreationExpression _new0(
      ArrayTypeName type, Iterable<? extends ExpressionCode> dims) {
    return build0(
        ImmutableCodeElement.builder()
            .withWord("new")
            .withTypeName(type.arrayCreationTypeName())
            .withCodeElements(dims(type, dims))
            .build()
    );
  }

  private static ArrayCreationExpression build0(ImmutableCodeElement delegate) {
    return new ArrayCreationExpressionImpl(delegate);
  }

  private static CodeElement dim(ExpressionCode dim) {
    return ImmutableCodeElement.builder()
        .withSpaceOff()
        .withOpenBracket()
        .withSpaceOff()
        .withCodeElement(dim)
        .withSpaceOff()
        .withCloseBracket()
        .build();
  }

  private static Iterable<CodeElement> dims(
      ArrayTypeName type, Iterable<? extends ExpressionCode> dims) {
    GrowableList<CodeElement> dimList = GrowableList.newList();
    Iterator<? extends ExpressionCode> it = dims.iterator();

    int dimension = type.dimension();
    for (int i = 0; i < dimension; i++) {
      ExpressionCode dim;
      if (it.hasNext()) {
        dim = it.next();
      } else {
        dim = EmptyExpression.empty();
      }
      dimList.add(dim(dim));
    }

    if (it.hasNext()) {
      throw new IllegalArgumentException("Too many dim expressions: array dimension=" + dimension);
    }

    return dimList;
  }

  @Override
  public final FieldAccess id(Identifier id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  public final FieldAccess id(String id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  protected final ArrayReferenceExpression selfArrayReferenceExpression() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final ConditionalAndExpression selfConditionalAndExpression() {
    return this;
  }

  @Override
  protected final Callee selfCallee() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final LeftHandSide selfLeftHandSide() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final MethodReferenceReferenceExpression selfMethodReferenceReferenceExpression() {
    return this;
  }

  @Override
  protected final MultiplicativeExpression selfMultiplicativeExpression() {
    return this;
  }

  @Override
  protected final PostfixExpression selfPostfixExpression() {
    return this;
  }

  @Override
  protected final RelationalExpression selfRelationalExpression() {
    return this;
  }

}
