/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import br.com.objectos.code.java.io.JavaFileImportSet;

public abstract class WildcardTypeName extends AbstractTypeName {

  static final WildcardTypeName UNBOUNDED = new Unbounded();

  WildcardTypeName() {}

  static WildcardTypeName extendsUnchecked(ReferenceTypeName bound) {
    return new Extends(bound);
  }

  static WildcardTypeName superUnchecked(ReferenceTypeName bound) {
    return new Super(bound);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    throw new UnsupportedOperationException();
  }

  public static class Extends extends WithBound {
    private Extends(ReferenceTypeName bound) {
      super(bound);
    }

    @Override
    public <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
      return visitor.visitWildcardTypeNameExtends(this, p);
    }

    @Override
    final String keyword() {
      return "extends";
    }
  }

  public static class Super extends WithBound {
    private Super(ReferenceTypeName bound) {
      super(bound);
    }

    @Override
    public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
      return visitor.visitWildcardTypeNameSuper(this, p);
    }

    @Override
    final String keyword() {
      return "super";
    }
  }

  public static class Unbounded extends WildcardTypeName {
    private Unbounded() {}

    @Override
    public final String acceptJavaFileImportSet(JavaFileImportSet set) {
      return toString();
    }

    @Override
    public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
      return visitor.visitWildcardTypeNameUnbounded(this, p);
    }

    @Override
    public final String toString() {
      return "?";
    }
  }

  private static abstract class WithBound extends WildcardTypeName {
    private final ReferenceTypeName bound;

    WithBound(ReferenceTypeName bound) {
      this.bound = bound;
    }

    @Override
    public final String acceptJavaFileImportSet(JavaFileImportSet set) {
      return toString(set.get(bound));
    }

    @Override
    public final boolean equals(Object obj) {
      if (obj == this) {
        return true;
      }
      if (!(obj instanceof WithBound)) {
        return false;
      }
      WithBound that = (WithBound) obj;
      return getClass().equals(that.getClass())
          && bound.equals(that.bound);
    }

    @Override
    public final int hashCode() {
      return bound.hashCode();
    }

    @Override
    public final String toString() {
      return toString(bound.toString());
    }

    abstract String keyword();

    private String toString(String type) {
      return "? " + keyword() + " " + type;
    }
  }

}