/*
 * Copyright 2015 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.sql.code;

import static com.google.common.collect.Lists.transform;

import java.util.List;

import javax.annotation.Generated;
import javax.lang.model.element.Modifier;

import br.com.objectos.sql.core.ColumnInfo;
import br.com.objectos.sql.core.ColumnInfoMap;
import br.com.objectos.sql.core.HasTableInfo;
import br.com.objectos.sql.core.TableInfo;
import br.com.objectos.sql.core.annotation.Table;
import br.com.objectos.way.code.CodeCanvasArtifact;
import br.com.objectos.way.code.CodeCanvasWriter;
import br.com.objectos.way.code.MethodInfo;
import br.com.objectos.way.code.SimpleTypeInfo;
import br.com.objectos.way.code.TypeInfo;
import br.com.objectos.way.core.auto.AutoFunctional;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeSpec;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class TableCode {

  private final TypeInfo typeInfo;
  private final MethodInfo methodInfo;
  private final TableInfo tableInfo;

  private final ClassName moduleClassName;
  private final ClassName tableClassName;

  private final List<ColumnCode> columnCodeList;

  private TableCode(TypeInfo typeInfo,
                    MethodInfo methodInfo,
                    TableInfo tableInfo,
                    ClassName moduleClassName,
                    ClassName tableClassName,
                    List<ColumnCode> columnCodeList) {
    this.typeInfo = typeInfo;
    this.methodInfo = methodInfo;
    this.tableInfo = tableInfo;
    this.moduleClassName = moduleClassName;
    this.tableClassName = tableClassName;
    this.columnCodeList = columnCodeList;
  }

  public static TableCode code(TypeInfo typeInfo, MethodInfo methodInfo, TableInfo tableInfo) {
    ClassName moduleClassName = typeInfo.toClassNamePrefix("___");
    final ClassName tableClassName = moduleClassName.peerClass(tableInfo.getClassName());

    List<ColumnCode> columnCodeList = tableInfo.transformColumnList(new Function<ColumnInfo, ColumnCode>() {
      @Override
      public ColumnCode apply(ColumnInfo columnInfo) {
        return ColumnCode.code(tableClassName, columnInfo);
      }
    });

    return new TableCode(
        typeInfo, methodInfo, tableInfo,
        moduleClassName, tableClassName,
        columnCodeList);
  }

  @AutoFunctional
  public boolean isValid() {
    SimpleTypeInfo returnTypeInfo = methodInfo.returnTypeInfo();
    if (!returnTypeInfo.isInfoOf(TableInfo.class)) {
      methodInfo.compilationError("Method must return a TableInfo instance.");
      return false;
    }

    if (!methodInfo.hasParameterInfoListSize(0)) {
      methodInfo.compilationError("Method must have zero arguments.");
      return false;
    }

    return true;
  }

  @AutoFunctional
  public CodeCanvasArtifact toCodeCanvasArtifact() {
    TypeSpec typeSpec = type();
    JavaFile javaFile = typeInfo.toJavaFile(typeSpec);
    return CodeCanvasWriter.forJavaFile(javaFile)
        .named(tableClassName)
        .toCodeCanvasArtifact();
  }

  @AutoFunctional
  public FieldSpec toSqlModuleFieldSpec() {
    String varName = typeInfo.getVarName();
    return FieldSpec.builder(TableInfo.class, methodInfo.name())
        .addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
        .initializer("$L.$L()", varName, methodInfo.name())
        .build();
  }

  @AutoFunctional
  public String schemaName() {
    return tableInfo.getSchemaName();
  }

  MethodSpec constructor() {
    return MethodSpec.constructorBuilder()
        .addModifiers(Modifier.PRIVATE)
        .build();
  }

  List<FieldSpec> staticField() {
    return ImmutableList.<FieldSpec> builder()
        .add(staticField0())
        .add(staticField1())
        .add(staticField2())
        .build();
  }

  MethodSpec staticMethod() {
    return MethodSpec.methodBuilder("get")
        .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
        .returns(tableClassName)
        .addStatement("return INSTANCE")
        .build();
  }

  TypeSpec type() {
    TypeSpec.Builder type = TypeSpec.classBuilder(tableInfo.getClassName())
        .addAnnotation(generatedAnnotationSpec())
        .addAnnotation(tableAnnotationSpec())
        .addModifiers(Modifier.PUBLIC, Modifier.FINAL)
        .addSuperinterface(HasTableInfo.class);

    type.addFields(staticField());

    type.addFields(transform(columnCodeList, ColumnCodeField.get()));

    type.addMethod(constructor());

    type.addMethod(staticMethod());

    type.addMethods(transform(columnCodeList, ColumnCodeMethod.get()));

    type.addMethod(tableInfoMethod());

    type.addTypes(transform(columnCodeList, ColumnCodeAnnotation.get()));

    type.addTypes(transform(columnCodeList, ColumnCodeInner.get()));

    return type.build();
  }

  private AnnotationSpec generatedAnnotationSpec() {
    return AnnotationSpec.builder(Generated.class)
        .addMember("value", "$S", "br.com.objectos.sql.compiler.SqlModuleProcessor")
        .build();
  }

  private FieldSpec staticField0() {
    return FieldSpec.builder(tableClassName, "INSTANCE")
        .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
        .initializer("new $T()", tableClassName)
        .build();
  }

  private FieldSpec staticField1() {
    return FieldSpec.builder(TableInfo.class, "TABLE")
        .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
        .initializer("$T.$L", moduleClassName, methodInfo.name())
        .build();
  }

  private FieldSpec staticField2() {
    return FieldSpec.builder(ColumnInfoMap.class, "COLUMN_INFO_MAP")
        .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
        .initializer("TABLE.toColumnInfoMap()")
        .build();
  }

  private AnnotationSpec tableAnnotationSpec() {
    return AnnotationSpec.builder(Table.class)
        .addMember("name", "\"" + tableInfo.name() + "\"")
        .build();
  }

  private MethodSpec tableInfoMethod() {
    return MethodSpec.methodBuilder("tableInfo")
        .addAnnotation(Override.class)
        .addModifiers(Modifier.PUBLIC)
        .returns(TableInfo.class)
        .addStatement("return TABLE")
        .build();
  }

}