package com.databricks.sdk.core;

import com.databricks.sdk.support.InternalApi;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@InternalApi
class ConfigAttributeAccessor {
  private final ConfigAttribute configAttribute;
  private final Field field;

  private final String configFileAttribute;

  public ConfigAttributeAccessor(ConfigAttribute configAttribute, Field field) {
    this.configAttribute = configAttribute;
    this.field = field;
    this.configFileAttribute = toSnakeCase(field.getName());
  }

  public String getName() {
    return configFileAttribute;
  }

  public String getEnvVariable() {
    return configAttribute.env();
  }

  public Boolean isSensitive() {
    return configAttribute.sensitive();
  }

  public String getEnv(Map<String, String> getEnv) {
    if (configAttribute.env().isEmpty()) {
      return "";
    }
    return getEnv.get(configAttribute.env());
  }

  public void setValueOnConfig(DatabricksConfig cfg, String value) throws IllegalAccessException {
    // Synchronize on field across all methods which alter its accessibility to ensure
    // multi threaded access of these objects (e.g. in the example of concurrent creation of
    // workspace clients or config resolution) are safe
    synchronized (field) {
      field.setAccessible(true);
      if (value == null || value.trim().isEmpty()) {
        return;
      }

      if (field.getType() == String.class) {
        field.set(cfg, value);
      } else if (field.getType() == int.class) {
        field.set(cfg, Integer.parseInt(value));
      } else if (field.getType() == Integer.class) {
        field.set(cfg, Integer.parseInt(value));
      } else if (field.getType() == boolean.class) {
        field.set(cfg, Boolean.parseBoolean(value));
      } else if (field.getType() == Boolean.class) {
        field.set(cfg, Boolean.parseBoolean(value));
      } else if (field.getType() == Duration.class) {
        int seconds = Integer.parseInt(value);
        field.set(cfg, seconds > 0 ? Duration.ofSeconds(seconds) : null);
      } else if (field.getType() == ProxyConfig.ProxyAuthType.class) {
        field.set(cfg, ProxyConfig.ProxyAuthType.valueOf(value));
      } else if (List.class.isAssignableFrom(field.getType())) {
        // Handle List<String> fields (e.g., scopes)
        // Parse comma-separated values from environment variable or config file.
        if (field.getGenericType() instanceof ParameterizedType) {
          ParameterizedType paramType = (ParameterizedType) field.getGenericType();
          if (paramType.getActualTypeArguments().length > 0
              && paramType.getActualTypeArguments()[0] == String.class) {
            // Split by comma, trim each value, and filter out empty strings
            List<String> list =
                Arrays.stream(value.trim().split(","))
                    .map(String::trim)
                    .filter(s -> !s.isEmpty())
                    .collect(Collectors.toList());
            field.set(cfg, list);
          }
        }
      }
      field.setAccessible(false);
    }
  }

  public Object getValueFromConfig(DatabricksConfig cfg) throws IllegalAccessException {
    synchronized (field) {
      field.setAccessible(true);
      Object value = field.get(cfg);
      field.setAccessible(false);
      return value;
    }
  }

  public String getAuthType() {
    return configAttribute.auth();
  }

  @Override
  public String toString() {
    String repr = configFileAttribute;
    if (!Objects.equals(configAttribute.env(), "")) {
      repr += "(env: " + configAttribute.env() + ")";
    }
    return repr;
  }

  public String getAsString(Object value) {
    if (value instanceof List) {
      List<?> list = (List<?>) value;
      return list.stream().map(Object::toString).collect(Collectors.joining(", ", "[", "]"));
    }
    return value.toString();
  }

  private String toSnakeCase(String name) {
    StringBuilder snakeCase = new StringBuilder();
    for (int i = 0; i < name.length(); i++) {
      char c = name.charAt(i);
      if (Character.isUpperCase(c)) {
        snakeCase.append("_").append(Character.toLowerCase(c));
      } else {
        snakeCase.append(c);
      }
    }
    return snakeCase.toString().toLowerCase();
  }
}
