/*
 * Decompiled with CFR 0.152.
 */
package org.jclouds.ssh;

import com.google.common.annotations.Beta;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteSource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Map;
import org.jclouds.crypto.Pems;
import org.jclouds.util.Strings2;

@Beta
public class SshKeys {
    public static RSAPublicKeySpec publicKeySpecFromOpenSSH(String idRsaPub) {
        try {
            return SshKeys.publicKeySpecFromOpenSSH(ByteSource.wrap(idRsaPub.getBytes(Charsets.UTF_8)));
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public static RSAPublicKeySpec publicKeySpecFromOpenSSH(ByteSource supplier) throws IOException {
        InputStream stream = supplier.openStream();
        Iterable<String> parts = Splitter.on(' ').split(Strings2.toStringAndClose(stream).trim());
        Preconditions.checkArgument(Iterables.size(parts) >= 2 && "ssh-rsa".equals(Iterables.get(parts, 0)), "bad format, should be: ssh-rsa AAAAB3...");
        stream = new ByteArrayInputStream(BaseEncoding.base64().decode(Iterables.get(parts, 1)));
        String marker = new String(SshKeys.readLengthFirst(stream));
        Preconditions.checkArgument("ssh-rsa".equals(marker), "looking for marker ssh-rsa but got %s", marker);
        BigInteger publicExponent = new BigInteger(SshKeys.readLengthFirst(stream));
        BigInteger modulus = new BigInteger(SshKeys.readLengthFirst(stream));
        return new RSAPublicKeySpec(modulus, publicExponent);
    }

    private static byte[] readLengthFirst(InputStream in) throws IOException {
        int byte1 = in.read();
        int byte2 = in.read();
        int byte3 = in.read();
        int byte4 = in.read();
        int length = (byte1 << 24) + (byte2 << 16) + (byte3 << 8) + (byte4 << 0);
        byte[] val = new byte[length];
        in.read(val, 0, length);
        return val;
    }

    public static KeyPair generateRsaKeyPair(KeyPairGenerator generator, SecureRandom rand) {
        generator.initialize(2048, rand);
        return generator.genKeyPair();
    }

    public static Map<String, String> generate() {
        try {
            return SshKeys.generate(KeyPairGenerator.getInstance("RSA"), new SecureRandom());
        }
        catch (NoSuchAlgorithmException e) {
            throw Throwables.propagate(e);
        }
    }

    public static Map<String, String> generate(KeyPairGenerator generator, SecureRandom rand) {
        KeyPair pair = SshKeys.generateRsaKeyPair(generator, rand);
        ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
        builder.put("public", SshKeys.encodeAsOpenSSH((RSAPublicKey)RSAPublicKey.class.cast(pair.getPublic())));
        builder.put("private", Pems.pem((PrivateKey)RSAPrivateKey.class.cast(pair.getPrivate())));
        return builder.build();
    }

    public static String encodeAsOpenSSH(RSAPublicKey key) {
        byte[] keyBlob = SshKeys.keyBlob(key.getPublicExponent(), key.getModulus());
        return "ssh-rsa " + BaseEncoding.base64().encode(keyBlob);
    }

    public static boolean privateKeyMatchesPublicKey(String privateKeyPEM, String publicKeyOpenSSH) {
        KeySpec privateKeySpec = Pems.privateKeySpec(privateKeyPEM);
        Preconditions.checkArgument(privateKeySpec instanceof RSAPrivateCrtKeySpec, "incorrect format expected RSAPrivateCrtKeySpec was %s", privateKeySpec);
        return SshKeys.privateKeyMatchesPublicKey((RSAPrivateCrtKeySpec)RSAPrivateCrtKeySpec.class.cast(privateKeySpec), SshKeys.publicKeySpecFromOpenSSH(publicKeyOpenSSH));
    }

    public static boolean privateKeyMatchesPublicKey(RSAPrivateCrtKeySpec privateKey, RSAPublicKeySpec publicKey) {
        return privateKey.getPublicExponent().equals(publicKey.getPublicExponent()) && privateKey.getModulus().equals(publicKey.getModulus());
    }

    public static boolean privateKeyHasFingerprint(RSAPrivateCrtKeySpec privateKey, String fingerprint) {
        return SshKeys.fingerprint(privateKey.getPublicExponent(), privateKey.getModulus()).equals(fingerprint);
    }

    public static boolean privateKeyHasFingerprint(String privateKeyPEM, String fingerprint) {
        KeySpec privateKeySpec = Pems.privateKeySpec(privateKeyPEM);
        Preconditions.checkArgument(privateKeySpec instanceof RSAPrivateCrtKeySpec, "incorrect format expected RSAPrivateCrtKeySpec was %s", privateKeySpec);
        return SshKeys.privateKeyHasFingerprint((RSAPrivateCrtKeySpec)RSAPrivateCrtKeySpec.class.cast(privateKeySpec), fingerprint);
    }

    public static String fingerprintPrivateKey(String privateKeyPEM) {
        KeySpec privateKeySpec = Pems.privateKeySpec(privateKeyPEM);
        Preconditions.checkArgument(privateKeySpec instanceof RSAPrivateCrtKeySpec, "incorrect format expected RSAPrivateCrtKeySpec was %s", privateKeySpec);
        RSAPrivateCrtKeySpec certKeySpec = (RSAPrivateCrtKeySpec)RSAPrivateCrtKeySpec.class.cast(privateKeySpec);
        return SshKeys.fingerprint(certKeySpec.getPublicExponent(), certKeySpec.getModulus());
    }

    public static String fingerprintPublicKey(String publicKeyOpenSSH) {
        RSAPublicKeySpec publicKeySpec = SshKeys.publicKeySpecFromOpenSSH(publicKeyOpenSSH);
        return SshKeys.fingerprint(publicKeySpec.getPublicExponent(), publicKeySpec.getModulus());
    }

    public static boolean privateKeyHasSha1(RSAPrivateCrtKeySpec privateKey, String fingerprint) {
        return SshKeys.sha1(privateKey).equals(fingerprint);
    }

    public static boolean privateKeyHasSha1(String privateKeyPEM, String sha1HexColonDelimited) {
        KeySpec privateKeySpec = Pems.privateKeySpec(privateKeyPEM);
        Preconditions.checkArgument(privateKeySpec instanceof RSAPrivateCrtKeySpec, "incorrect format expected RSAPrivateCrtKeySpec was %s", privateKeySpec);
        return SshKeys.privateKeyHasSha1((RSAPrivateCrtKeySpec)RSAPrivateCrtKeySpec.class.cast(privateKeySpec), sha1HexColonDelimited);
    }

    public static String sha1PrivateKey(String privateKeyPEM) {
        KeySpec privateKeySpec = Pems.privateKeySpec(privateKeyPEM);
        Preconditions.checkArgument(privateKeySpec instanceof RSAPrivateCrtKeySpec, "incorrect format expected RSAPrivateCrtKeySpec was %s", privateKeySpec);
        RSAPrivateCrtKeySpec certKeySpec = (RSAPrivateCrtKeySpec)RSAPrivateCrtKeySpec.class.cast(privateKeySpec);
        return SshKeys.sha1(certKeySpec);
    }

    public static String sha1(RSAPrivateCrtKeySpec privateKey) {
        try {
            byte[] encodedKey = KeyFactory.getInstance("RSA").generatePrivate(privateKey).getEncoded();
            return SshKeys.hexColonDelimited(Hashing.sha1().hashBytes(encodedKey));
        }
        catch (InvalidKeySpecException e) {
            throw Throwables.propagate(e);
        }
        catch (NoSuchAlgorithmException e) {
            throw Throwables.propagate(e);
        }
    }

    public static boolean publicKeyHasFingerprint(RSAPublicKeySpec publicKey, String fingerprint) {
        return SshKeys.fingerprint(publicKey.getPublicExponent(), publicKey.getModulus()).equals(fingerprint);
    }

    public static boolean publicKeyHasFingerprint(String publicKeyOpenSSH, String fingerprint) {
        return SshKeys.publicKeyHasFingerprint(SshKeys.publicKeySpecFromOpenSSH(publicKeyOpenSSH), fingerprint);
    }

    public static String fingerprint(BigInteger publicExponent, BigInteger modulus) {
        byte[] keyBlob = SshKeys.keyBlob(publicExponent, modulus);
        return SshKeys.hexColonDelimited(Hashing.md5().hashBytes(keyBlob));
    }

    private static String hexColonDelimited(HashCode hc) {
        return Joiner.on(':').join(Splitter.fixedLength(2).split(BaseEncoding.base16().lowerCase().encode(hc.asBytes())));
    }

    private static byte[] keyBlob(BigInteger publicExponent, BigInteger modulus) {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            SshKeys.writeLengthFirst("ssh-rsa".getBytes(), out);
            SshKeys.writeLengthFirst(publicExponent.toByteArray(), out);
            SshKeys.writeLengthFirst(modulus.toByteArray(), out);
            return out.toByteArray();
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private static void writeLengthFirst(byte[] array, ByteArrayOutputStream out) throws IOException {
        out.write(array.length >>> 24 & 0xFF);
        out.write(array.length >>> 16 & 0xFF);
        out.write(array.length >>> 8 & 0xFF);
        out.write(array.length >>> 0 & 0xFF);
        if (array.length == 1 && array[0] == 0) {
            out.write(new byte[0]);
        } else {
            out.write(array);
        }
    }
}

