/*
 * Decompiled with CFR 0.152.
 */
package com.eventstore.dbclient;

import com.eventstore.dbclient.ConnectionSettingsBuilder;
import com.eventstore.dbclient.Consts;
import com.eventstore.dbclient.Endpoint;
import com.eventstore.dbclient.EventStoreDBClientSettings;
import com.eventstore.dbclient.NodePreference;
import com.eventstore.dbclient.ParseError;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventStoreDBConnectionString {
    private final Logger logger = LoggerFactory.getLogger(EventStoreDBConnectionString.class);
    private int position = 0;
    private int nextPosition = 0;
    private String connectionString;
    private final ConnectionSettingsBuilder settings = EventStoreDBClientSettings.builder();
    private final List<String> notCurrentlySupported = Arrays.asList("maxDiscoverAttempts", "discoveryInterval", "gossipTimeout", "throwOnAppendFailure", "defaultCredentials");
    private HashMap<String, String> LowerToKey = new HashMap<String, String>(){
        {
            this.put("dnsdiscover", "dnsDiscover");
            this.put("maxdiscoverattempts", "maxDiscoverAttempts");
            this.put("discoveryinterval", "discoveryInterval");
            this.put("gossiptimeout", "gossipTimeout");
            this.put("nodepreference", "nodePreference");
            this.put("tls", "tls");
            this.put("tlsverifycert", "tlsVerifyCert");
            this.put("throwonappendfailure", "throwOnAppendFailure");
            this.put("keepalivetimeout", "keepAliveTimeout");
            this.put("keepaliveinterval", "keepAliveInterval");
        }
    };
    private static final String[] NODE_PREFERENCE_VALUES = new String[]{"leader", "follower", "readonlyreplica", "random"};

    public static EventStoreDBClientSettings parse(String connectionString) throws ParseError {
        return new EventStoreDBConnectionString().parseConnectionString(connectionString);
    }

    public static EventStoreDBClientSettings parseOrThrow(String connectionString) {
        try {
            return EventStoreDBConnectionString.parse(connectionString);
        }
        catch (ParseError e) {
            throw new RuntimeException(e);
        }
    }

    private String getRemaining() {
        return this.connectionString.substring(this.position);
    }

    private EventStoreDBClientSettings parseConnectionString(String connectionString) throws ParseError {
        this.connectionString = connectionString.trim().replaceAll("/+$", "");
        return this.parseProtocol();
    }

    private EventStoreDBClientSettings parseProtocol() throws ParseError {
        this.position = this.nextPosition;
        String expected = "esdb:// or esdb+discover://";
        String pattern = "^(?<protocol>[^:]+)://";
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(this.getRemaining());
        boolean found = m.find();
        if (found && !m.group("protocol").isEmpty()) {
            this.nextPosition += m.end();
            switch (m.group("protocol")) {
                case "esdb": {
                    this.settings.dnsDiscover(false);
                    return this.parseCredentials();
                }
                case "esdb+discover": {
                    this.settings.dnsDiscover(true);
                    return this.parseCredentials();
                }
            }
        }
        throw new ParseError(this.connectionString, this.position, this.nextPosition, expected);
    }

    private EventStoreDBClientSettings parseCredentials() throws ParseError {
        this.position = this.nextPosition;
        String expected = "<URL encoded username>:<Url encoded password>";
        String pattern = "^(?:(?<credentials>[^:]+:[^@]+)@)";
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(this.getRemaining());
        boolean found = m.find();
        if (!found) {
            return this.parseHosts(true);
        }
        if (!m.group("credentials").isEmpty()) {
            try {
                this.nextPosition += m.end();
                String[] credentials = m.group("credentials").split(":");
                String username = URLDecoder.decode(credentials[0], "utf-8");
                String password = URLDecoder.decode(credentials[1], "utf-8");
                this.settings.defaultCredentials(username, password);
            }
            catch (UnsupportedEncodingException e) {
                throw new ParseError(this.connectionString, this.position, this.nextPosition, expected);
            }
            return this.parseHosts(true);
        }
        throw new ParseError(this.connectionString, this.position, this.nextPosition, expected);
    }

    private EventStoreDBClientSettings parseHosts(boolean mustMatch) throws ParseError {
        this.position = this.nextPosition;
        String expected = "<URL encoded username>:<Url encoded password>";
        String pattern = "^(?:(?<host>[^$+!?*'(),;\\[\\]{}|\"%~#<>=&/]+)[,/]?)";
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(this.getRemaining());
        boolean found = m.find();
        if (!found && mustMatch) {
            throw new ParseError(this.connectionString, this.position, this.nextPosition, expected);
        }
        if (found && !m.group("host").isEmpty()) {
            String rawPort;
            this.nextPosition += m.end();
            String[] hostParts = m.group("host").split(":");
            String address = hostParts[0];
            String string = rawPort = hostParts.length > 1 ? hostParts[1] : "2113";
            if (hostParts.length > 2) {
                throw new ParseError(this.connectionString, this.position + (address + ":" + rawPort).length(), this.nextPosition, ", or ?key=value");
            }
            try {
                int port = Integer.parseInt(rawPort);
                Endpoint host = new Endpoint(address, port);
                this.settings.addHost(host);
            }
            catch (NumberFormatException e) {
                throw new ParseError(this.connectionString, this.position + address.length(), this.nextPosition, "port number");
            }
            return this.parseHosts(false);
        }
        return this.parseSearchParams(true);
    }

    public static Optional<NodePreference> parseNodePreference(String value) {
        switch (value.toLowerCase()) {
            case "leader": {
                return Optional.of(NodePreference.LEADER);
            }
            case "follower": {
                return Optional.of(NodePreference.FOLLOWER);
            }
            case "readonlyreplica": {
                return Optional.of(NodePreference.READ_ONLY_REPLICA);
            }
            case "random": {
                return Optional.of(NodePreference.RANDOM);
            }
        }
        return Optional.empty();
    }

    private EventStoreDBClientSettings parseSearchParams(boolean first) throws ParseError {
        this.position = this.nextPosition;
        if (this.position == this.connectionString.length()) {
            return this.settings.buildConnectionSettings();
        }
        String expected = first ? "?key=value" : "&key=value";
        String pattern = "^(?:" + (first ? "\\?" : "&") + "(?<key>[^=]+)=(?<value>[^&?]+))";
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(this.getRemaining());
        boolean found = m.find();
        if (!found || m.group("key").isEmpty() || m.group("value").isEmpty()) {
            throw new ParseError(this.connectionString, this.position, this.nextPosition, expected);
        }
        this.nextPosition += m.end();
        String rawKey = m.group("key");
        String key = this.LowerToKey.getOrDefault(rawKey.toLowerCase(Locale.ROOT), rawKey);
        String value = m.group("value");
        int keyPosition = this.position + ("&" + key + "=").length();
        if (this.notCurrentlySupported.contains(key)) {
            this.logger.warn("{} is not currently supported by this client, and will have no effect.", (Object)key);
        }
        switch (key) {
            case "nodePreference": {
                Optional<NodePreference> preference = EventStoreDBConnectionString.parseNodePreference(value);
                if (preference.isPresent()) {
                    this.settings.nodePreference(preference.get());
                    break;
                }
                throw new ParseError(this.connectionString, keyPosition, this.nextPosition, Arrays.toString(NODE_PREFERENCE_VALUES));
            }
            case "maxDiscoverAttempts": {
                try {
                    int parsedValue = Integer.parseInt(value);
                    this.settings.maxDiscoverAttempts(parsedValue);
                    break;
                }
                catch (NumberFormatException e) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "integer");
                }
            }
            case "discoveryInterval": {
                try {
                    int parsedValue = Integer.parseInt(value);
                    this.settings.discoveryInterval(parsedValue);
                    break;
                }
                catch (NumberFormatException e) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "integer");
                }
            }
            case "gossipTimeout": {
                try {
                    int parsedValue = Integer.parseInt(value);
                    this.settings.gossipTimeout(parsedValue);
                    break;
                }
                catch (NumberFormatException e) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "integer");
                }
            }
            case "dnsDiscover": {
                if (!value.equals("true") && !value.equals("false")) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "true or false");
                }
                this.settings.dnsDiscover(value.equals("true"));
                break;
            }
            case "tls": {
                if (!value.equals("true") && !value.equals("false")) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "true or false");
                }
                this.settings.tls(value.equals("true"));
                break;
            }
            case "tlsVerifyCert": {
                if (!value.equals("true") && !value.equals("false")) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "true or false");
                }
                this.settings.tlsVerifyCert(value.equals("true"));
                break;
            }
            case "throwOnAppendFailure": {
                if (!value.equals("true") && !value.equals("false")) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "true or false");
                }
                this.settings.throwOnAppendFailure(value.equals("true"));
                break;
            }
            case "keepAliveTimeout": {
                try {
                    long parsedValue = Long.parseLong(value);
                    if (parsedValue >= 0L && parsedValue < Consts.DEFAULT_KEEP_ALIVE_TIMEOUT_IN_MS) {
                        this.logger.warn("Specified keepAliveTimeout of {} is less than recommended {}", (Object)parsedValue, (Object)Consts.DEFAULT_KEEP_ALIVE_TIMEOUT_IN_MS);
                    }
                    if (parsedValue < -1L) {
                        this.logger.error("Invalid keepAliveTimeout of {}. Please provide a positive integer, or -1 to disable.", (Object)parsedValue);
                        throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "positive integer");
                    }
                    if (parsedValue == -1L) {
                        parsedValue = Long.MAX_VALUE;
                    }
                    this.settings.keepAliveTimeout(parsedValue);
                    break;
                }
                catch (NumberFormatException e) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "integer");
                }
            }
            case "keepAliveInterval": {
                try {
                    long parsedValue = Long.parseLong(value);
                    if (parsedValue >= 0L && parsedValue < Consts.DEFAULT_KEEP_ALIVE_INTERVAL_IN_MS) {
                        this.logger.warn("Specified keepAliveInterval of {} is less than recommended {}", (Object)parsedValue, (Object)Consts.DEFAULT_KEEP_ALIVE_INTERVAL_IN_MS);
                    }
                    if (parsedValue < -1L) {
                        this.logger.error("Invalid keepAliveInterval of {}. Please provide a positive integer, or -1 to disable.", (Object)parsedValue);
                        throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "positive integer");
                    }
                    if (parsedValue == -1L) {
                        parsedValue = Long.MAX_VALUE;
                    }
                    this.settings.keepAliveInterval(parsedValue);
                    break;
                }
                catch (NumberFormatException e) {
                    throw new ParseError(this.connectionString, keyPosition, this.nextPosition, "integer");
                }
            }
            default: {
                this.logger.warn("Unknown option {}, setting will be ignored.", (Object)key);
            }
        }
        return this.parseSearchParams(false);
    }
}

