/*
 * Decompiled with CFR 0.152.
 */
package com.eventstore.dbclient;

import com.eventstore.dbclient.ConnectionShutdownException;
import com.eventstore.dbclient.Endpoint;
import com.eventstore.dbclient.EventStoreDBClientSettings;
import com.eventstore.dbclient.NotLeaderException;
import com.eventstore.dbclient.Timeouts;
import io.grpc.ManagedChannel;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.SslContext;
import java.util.ArrayList;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class GrpcClient {
    protected final EventStoreDBClientSettings settings;
    protected final SslContext sslContext;
    private final Logger logger = LoggerFactory.getLogger(GrpcClient.class);
    private final LinkedBlockingQueue<Msg> messages;
    protected ManagedChannel channel;
    protected Exception lastException;
    protected UUID currentChannelId;
    protected volatile boolean shutdown = false;

    protected GrpcClient(EventStoreDBClientSettings settings, SslContext sslContext) {
        this.settings = settings;
        this.sslContext = sslContext;
        this.messages = new LinkedBlockingQueue();
        this.currentChannelId = UUID.randomUUID();
    }

    protected void startConnectionLoop() {
        this.pushMsg(new CreateChannel(this.currentChannelId));
        CompletableFuture.runAsync(this::messageLoop, this.createConnectionLoopExecutor());
    }

    protected Executor createConnectionLoopExecutor() {
        return Executors.newSingleThreadExecutor(r -> {
            Thread thread = new Thread(r, "esdb-client-" + this.currentChannelId);
            thread.setDaemon(true);
            return thread;
        });
    }

    protected abstract boolean doConnect();

    protected void pushMsg(Msg msg) {
        try {
            if (this.shutdown) {
                if (msg instanceof RunWorkItem) {
                    RunWorkItem args = (RunWorkItem)msg;
                    args.reportError(new ConnectionShutdownException());
                }
                if (msg instanceof Shutdown) {
                    ((Shutdown)msg).completed.accept(42);
                }
                return;
            }
            this.messages.put(msg);
        }
        catch (InterruptedException e) {
            this.logger.error("Unexpected exception occurred when pushing a new message", (Throwable)e);
        }
    }

    public <A> CompletableFuture<A> run(Function<ManagedChannel, CompletableFuture<A>> action) {
        CompletableFuture result = new CompletableFuture();
        GrpcClient self = this;
        this.pushMsg(new RunWorkItem((id, channel, fatalError) -> {
            if (fatalError != null) {
                result.completeExceptionally(fatalError);
                return;
            }
            ((CompletableFuture)action.apply(channel)).whenComplete((outcome, error) -> {
                StatusRuntimeException ex;
                if (outcome != null) {
                    result.complete(outcome);
                    return;
                }
                if (error instanceof NotLeaderException) {
                    NotLeaderException ex2 = (NotLeaderException)error;
                    result.completeExceptionally(ex2);
                    self.pushMsg(new CreateChannel(id, ex2.getLeaderEndpoint()));
                    return;
                }
                if (error instanceof StatusRuntimeException && ((ex = (StatusRuntimeException)((Object)((Object)error))).getStatus().getCode().equals((Object)Status.Code.UNAVAILABLE) || ex.getStatus().getCode().equals((Object)Status.Code.ABORTED))) {
                    self.pushMsg(new CreateChannel(id));
                }
                result.completeExceptionally((Throwable)error);
            });
        }));
        return result;
    }

    private boolean discover(UUID previousId, Optional<Endpoint> candidate) {
        long attempts = 1L;
        if (!this.currentChannelId.equals(previousId)) {
            return true;
        }
        if (candidate.isPresent()) {
            this.channel = this.createChannel(candidate.get());
            this.currentChannelId = UUID.randomUUID();
            return true;
        }
        while (true) {
            this.logger.debug("Start connection attempt ({}/{})", (Object)attempts, (Object)this.settings.getMaxDiscoverAttempts());
            if (this.doConnect()) {
                this.currentChannelId = UUID.randomUUID();
                this.logger.info("Connection created successfully");
                return true;
            }
            if (++attempts > (long)this.settings.getMaxDiscoverAttempts()) {
                this.logger.error("Maximum discovery attempt count reached: {}", (Object)this.settings.getMaxDiscoverAttempts());
                return false;
            }
            this.logger.warn("Unable to find a node. Retrying... ({}/{})", (Object)attempts, (Object)this.settings.getMaxDiscoverAttempts());
            this.sleep(this.settings.getDiscoveryInterval());
        }
    }

    protected void sleep(long millis) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException e) {
            this.logger.error("Thread is interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private boolean handleMsg(Msg msg) {
        boolean result = false;
        if (msg instanceof CreateChannel) {
            if (!this.shutdown) {
                CreateChannel args = (CreateChannel)msg;
                result = this.discover(args.previousId, args.channel);
            } else {
                this.logger.warn("Channel creation request ignored, the connection is already closed");
            }
        } else if (msg instanceof RunWorkItem) {
            RunWorkItem args = (RunWorkItem)msg;
            if (this.shutdown) {
                Exception e = this.lastException != null ? this.lastException : new ConnectionShutdownException();
                this.logger.warn("Receive an command request but the connection is already closed", (Throwable)e);
                args.item.accept(null, null, e);
            } else if (this.channel == null) {
                try {
                    this.messages.put(new RunWorkItem(args.item));
                    this.logger.debug("Channel is not resolved yet, parking current work item");
                }
                catch (InterruptedException e) {
                    this.logger.error("Exception occurred when parking a work item", (Throwable)e);
                    args.item.accept(null, null, e);
                }
            } else {
                args.item.accept(this.currentChannelId, this.channel, null);
            }
            result = true;
        } else if (msg instanceof Shutdown) {
            if (!this.shutdown) {
                this.logger.info("Received a shutdown request, closing...");
                this.closeConnection();
                result = false;
                this.logger.info("Connection was closed successfully");
            } else {
                this.logger.info("Shutdown request ignored, connection is already closed");
            }
        }
        return result;
    }

    private void messageLoop() {
        Consumer<Object> shutdownCompleted = null;
        try {
            Msg msg;
            do {
                msg = this.messages.take();
                this.logger.debug("Current msg: {}", (Object)msg);
            } while (this.handleMsg(msg));
            this.shutdown = true;
            if (msg instanceof Shutdown) {
                shutdownCompleted = ((Shutdown)msg).completed;
            }
        }
        catch (InterruptedException e) {
            this.lastException = e;
            this.shutdown = true;
        }
        this.logger.debug("Draining pending requests...");
        ArrayList msgs = new ArrayList();
        this.messages.drainTo(msgs);
        for (Msg msg : msgs) {
            this.handleMsg(msg);
        }
        if (shutdownCompleted != null) {
            shutdownCompleted.accept(42);
        }
        this.logger.debug("Drainage completed successfully");
    }

    private void closeConnection() {
        if (this.channel != null) {
            try {
                this.channel.shutdown().awaitTermination(Timeouts.DEFAULT.shutdownTimeout, Timeouts.DEFAULT.shutdownTimeoutUnit);
            }
            catch (InterruptedException e) {
                this.logger.error("Error when closing gRPC channel", (Throwable)e);
            }
            finally {
                this.channel = null;
            }
        }
    }

    protected ManagedChannel createChannel(Endpoint endpoint) {
        NettyChannelBuilder builder = NettyChannelBuilder.forAddress((String)endpoint.getHostname(), (int)endpoint.getPort());
        if (this.sslContext == null) {
            builder.usePlaintext();
        } else {
            builder.sslContext(this.sslContext);
        }
        if (this.settings.getKeepAliveTimeout() <= 0L) {
            builder.keepAliveTimeout(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
        } else {
            builder.keepAliveTimeout(this.settings.getKeepAliveTimeout(), TimeUnit.MILLISECONDS);
        }
        if (this.settings.getKeepAliveInterval() <= 0L) {
            builder.keepAliveTime(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
        } else {
            builder.keepAliveTime(this.settings.getKeepAliveInterval(), TimeUnit.MILLISECONDS);
        }
        return builder.build();
    }

    public void shutdown() throws ExecutionException, InterruptedException {
        CompletableFuture completion = new CompletableFuture();
        this.pushMsg(new Shutdown(completion::complete));
        completion.get();
    }

    class Shutdown
    implements Msg {
        final Consumer<Object> completed;

        Shutdown(Consumer<Object> completed) {
            this.completed = completed;
        }
    }

    class RunWorkItem
    implements Msg {
        final WorkItem item;

        RunWorkItem(WorkItem item) {
            this.item = item;
        }

        void reportError(Exception e) {
            this.item.accept(null, null, e);
        }
    }

    class CreateChannel
    implements Msg {
        final Optional<Endpoint> channel;
        final UUID previousId;

        CreateChannel(UUID previousId) {
            this.channel = Optional.empty();
            this.previousId = previousId;
        }

        CreateChannel(UUID previousId, Endpoint endpoint) {
            this.channel = Optional.of(endpoint);
            this.previousId = previousId;
        }
    }

    private static interface Msg {
    }

    @FunctionalInterface
    static interface WorkItem {
        public void accept(UUID var1, ManagedChannel var2, Exception var3);
    }
}

