package com.flutterwave.services;

import com.flutterwave.bean.PreAuthorizationRequest;
import com.flutterwave.bean.Response;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.Optional;

import static com.flutterwave.bean.ChargeTypes.CARD;
import static com.flutterwave.bean.PreAuthCommands.*;

/**
 * Manage preauth attempts on Card and PayPal transactions. You can capture, void or refund preauth transactions.
 * @author Cleopatra Douglas
 */
public class PreAuthorization extends Charge{

    /**
     * Create a lien on the customer's account for the transaction amount. The hold on the funds only lasts for seven working days.
     * @param preAuthorizationRequest bean
     * @return Response
     */
    public Response runTransaction(PreAuthorizationRequest preAuthorizationRequest){
        return runTransaction(preAuthorizationRequest.toString(), CARD, true, Optional.empty());
    }

    /**
     * capture the payment of an existing but uncaptured charge.
     * This is the final step of a two-part payment flow. Where you first need to create the charge.
     * @param flw_ref String This is the data.flw_ref returned in the charge response.
     * @param amount BigDecimal This is the amount to be captured. This allows you to capture an amount less than the original amount authorised on the card.
     * @return Response
     */
    public Response runCaptureTransaction(String flw_ref, BigDecimal amount){
        return runTransaction(new JSONObject().put("amount", amount).toString(), CARD, true, Optional.of("/"+flw_ref+"/" + capture));
    }

    /**
     * Capture the payment of a previously uncaptured PayPal charge. Like Card charge capture, This is also the second leg of a two-part flow.
     * @param flw_ref String This is a unique reference internally generated by Flutterwave. It is returned as data.flw_ref in the charge response.
     * @return Response
     */
    public Response runPaypalCaptureTransaction(String flw_ref){
        return runTransaction(new JSONObject().put("flw_ref", flw_ref).toString(), CARD, true, Optional.of("/paypal-capture"));
    }

    /**
     * Voids the payment of a captured charge. This releases the hold on the previously captured funds
     * @param flw_ref String This is a unique reference internally generated by Flutterwave. It is returned as data.flw_ref in the charge response.
     * @return Response
     */
    public Response runVoidTransaction(String flw_ref){
        return runTransaction(null, CARD, true, Optional.of("/"+flw_ref+"/" + voId.toString().toLowerCase()));
    }

    /**
     * Release the hold on the PayPal funds i.e. if value was not given for the service, the merchant would typically be required to void the transaction.
     * @param flw_ref String This is the data.flw_ref returned in the charge response.
     * @return Response
     */
    public Response runPaypalVoidTransaction(String flw_ref){
        return runTransaction(null, CARD, true, Optional.of("/"+flw_ref+"/paypal-void"));
    }

    /**
     * Refunding a transaction would return the charged amount for an existing, unrefunded transaction.
     * Funds will be refunded to the debit or credit cards charged originally.
     * @param flw_ref String This is a unique reference internally generated by Flutterwave. It is returned as data.flw_ref in the charge response.
     * @param amount BigDecimal This is the amount to be refunded.
     * @return Response
     */
    public Response runRefundTransaction(String flw_ref, BigDecimal amount){
        return runTransaction(new JSONObject().put("amount", amount).toString(), CARD, true, Optional.of("/"+flw_ref+"/" + refund));
    }

}
