package com.glideapi; 

import com.glideapi.services.*;
import com.glideapi.Types.*;
import io.github.cdimascio.dotenv.Dotenv;

public class GlideClient {
    private final GlideSdkSettings settings;
    
    public final TelcoFinderClient telcoFinder;
    public final MagicAuthClient magicAuth;
    public final SimSwapClient simSwap;
    public final NumberVerifyClient numberVerify;

    public GlideClient(GlideSdkSettings settings) {
        Dotenv dotenv = Dotenv.configure().ignoreIfMissing().load();
        GlideSdkSettings defaults = new GlideSdkSettings(
            dotenv.get("GLIDE_CLIENT_ID", ""),
            dotenv.get("GLIDE_CLIENT_SECRET", ""),
            dotenv.get("GLIDE_REDIRECT_URI", ""),
            new GlideSdkSettings.Internal(
                dotenv.get("GLIDE_AUTH_BASE_URL", "https://oidc.gateway-x.io"),
                dotenv.get("GLIDE_API_BASE_URL", "https://api.gateway-x.io")
            )
        );

        this.settings = mergeSettings(defaults, settings);

        if (this.settings.getClientId() == null || this.settings.getClientId().isEmpty()) {
            throw new IllegalArgumentException("clientId is required");
        }

        if (this.settings.getClientSecret() == null || this.settings.getClientSecret().isEmpty()) {
            throw new IllegalArgumentException("clientSecret is required");
        }

        if (this.settings.getInternal().getAuthBaseUrl() == null || this.settings.getInternal().getAuthBaseUrl().isEmpty()) {
            throw new IllegalArgumentException("internal.authBaseUrl is unset");
        }

        this.telcoFinder = new TelcoFinderClient(this.settings);
        this.magicAuth = new MagicAuthClient(this.settings);
        this.simSwap = new SimSwapClient(this.settings);
        this.numberVerify = new NumberVerifyClient(this.settings);
    }

    public GlideClient(String clientId, String clientSecret) {
        this(new GlideSdkSettings(clientId, clientSecret, "", new GlideSdkSettings.Internal("https://oidc.gateway-x.io", "https://api.gateway-x.io")));
    }

    public GlideClient() {
        this(new GlideSdkSettings());
    }

    private GlideSdkSettings mergeSettings(GlideSdkSettings defaults, GlideSdkSettings userSettings) {
        return new GlideSdkSettings(
            userSettings.getClientId() != null ? userSettings.getClientId() : defaults.getClientId(),
            userSettings.getClientSecret() != null ? userSettings.getClientSecret() : defaults.getClientSecret(),
            userSettings.getRedirectUri() != null ? userSettings.getRedirectUri() : defaults.getRedirectUri(),
            new GlideSdkSettings.Internal(
                userSettings.getInternal().getAuthBaseUrl() != null ? userSettings.getInternal().getAuthBaseUrl() : defaults.getInternal().getAuthBaseUrl(),
                userSettings.getInternal().getApiBaseUrl() != null ? userSettings.getInternal().getApiBaseUrl() : defaults.getInternal().getApiBaseUrl()
            )
        );
    }
}
