package com.glideapi.services;

import com.glideapi.Types.*;
import com.glideapi.Utils;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.*;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;

public class MagicAuthClient {
    private final GlideSdkSettings settings;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private Session session = null;

    public MagicAuthClient(GlideSdkSettings settings) {
        this.settings = settings;
    }

    public enum FallbackVerificationChannel {
        SMS,
        EMAIL,
        NO_FALLBACK
    }

    public static class BaseMagicAuthStartProps {
        public FallbackVerificationChannel fallbackChannel;
    }

    public static class MagicAuthStartPropsEmail extends BaseMagicAuthStartProps {
        public String email;
    }

    public static class MagicAuthStartPropsPhone extends BaseMagicAuthStartProps {
        public String phoneNumber;
        public String redirectUrl;
        public String state;

        public MagicAuthStartPropsPhone(String phoneNumber) {
            this.phoneNumber = phoneNumber;
            this.fallbackChannel = FallbackVerificationChannel.NO_FALLBACK;
        }

        public MagicAuthStartPropsPhone() {
            this.fallbackChannel = FallbackVerificationChannel.NO_FALLBACK;
        }
    }

    public static class MagicAuthStartCodeResponse {
        public String type; // "EMAIL" or "SMS"
    }

    public static class MagicAuthStartMagicResponse {
        public String type; // "MAGIC"
        public String authUrl;
    }

    public static class MagicAuthStartResponse {
        public String type;
        public String authUrl;
        public String flatAuthUrl;
        public String operatorId;
    }

    public static class MagicAuthVerifyProps {
        public String code;
        public String phoneNumber;
        public String email;
        public String token;
    
        // Private constructor to force use of static factory methods
        private MagicAuthVerifyProps() {}
    
        public static MagicAuthVerifyProps createEmailProps(String email, String code) {
            MagicAuthVerifyProps props = new MagicAuthVerifyProps();
            props.email = email;
            props.code = code;
            return props;
        }
    
        public static MagicAuthVerifyProps createPhoneProps(String phoneNumber, String code) {
            MagicAuthVerifyProps props = new MagicAuthVerifyProps();
            props.phoneNumber = phoneNumber;
            props.code = code;
            return props;
        }
    
        public static MagicAuthVerifyProps createMagicProps(String phoneNumber, String token) {
            MagicAuthVerifyProps props = new MagicAuthVerifyProps();
            props.phoneNumber = phoneNumber;
            props.token = token;
            return props;
        }
    }

    public static class MagicAuthCheckResponse {
        public boolean verified;
    }

    public MagicAuthStartResponse startAuth(BaseMagicAuthStartProps props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
        }
        
        if(conf.getSessionIdentifier() != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide start");
        }
        Session session = conf.getSession() != null ? conf.getSession() : getSession();
        Map<String, Object> data = new HashMap<>();
        if (props instanceof MagicAuthStartPropsPhone) {
            MagicAuthStartPropsPhone phoneProps = (MagicAuthStartPropsPhone) props;
            data.put("phoneNumber", phoneProps.phoneNumber);
            if (phoneProps.redirectUrl != null) data.put("redirectUrl", phoneProps.redirectUrl);
            if (phoneProps.state != null) data.put("state", phoneProps.state);
        } else if (props instanceof MagicAuthStartPropsEmail) {
            MagicAuthStartPropsEmail emailProps = (MagicAuthStartPropsEmail) props;
            data.put("email", emailProps.email);
        }
        if (props.fallbackChannel != null) {
            data.put("fallbackChannel", props.fallbackChannel.name());
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/start"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(data)))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            throw new RuntimeException("Failed to start auth: " + response.body());
        }

        MagicAuthStartResponse result = objectMapper.readValue(response.body(), MagicAuthStartResponse.class);
        if(conf.getSessionIdentifier() != null && result.operatorId != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide verificationStartRes", result.operatorId);
        }
        return result;
    }

    public MagicAuthCheckResponse verifyAuth(MagicAuthVerifyProps props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = conf.getSession() != null ? conf.getSession() : getSession();
        Map<String, String> data = new HashMap<>();

        if (props.phoneNumber != null && props.token != null) {
            // MagicAuthVerifyMagicProps
            data.put("phoneNumber", props.phoneNumber);
            data.put("token", props.token);
        } else if (props.email != null && props.code != null) {
            // MagicAuthVerifyEmailProps
            data.put("email", props.email);
            data.put("code", props.code);
        } else if (props.phoneNumber != null && props.code != null) {
            // MagicAuthVerifyPhoneProps
            data.put("phoneNumber", props.phoneNumber);
            data.put("code", props.code);
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/check"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(data)))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            throw new RuntimeException("Failed to verify auth: " + response.body());
        }
        if (conf.getSessionIdentifier() != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide success");
        }
        MagicAuthCheckResponse result = objectMapper.readValue(response.body(), MagicAuthCheckResponse.class);
        if (conf.getSessionIdentifier() != null) {
            if (result.verified) {
                this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide verified");
            } else {
                this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide unverified");
            }
        }
        return result;
    }

    private void reportMagicAuthMetric(String sessionId, String metricName) {
        reportMagicAuthMetric(sessionId, metricName, null);
    }

    private void reportMagicAuthMetric(String sessionId, String metricName, String operator) {
        Utils.reportMetric(new Utils.MetricInfo(
            new Date(),
            sessionId,
            metricName,
            "magic-auth",
            settings.getClientId(),
            operator
        ));
    }

    private Session getSession() throws Exception {
        if (this.session != null && this.session.getExpiresAt() > System.currentTimeMillis() + (60 * 1000) && 
            this.session.getScopes().contains("magic-auth")) {
            return this.session;
        }
        Session session = generateNewSession();
        this.session = session;
        return session;
    }

    private Session generateNewSession() throws Exception {
        if (settings.getClientId() == null || settings.getClientSecret() == null) {
            throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
            .header("Content-Type", "application/x-www-form-urlencoded")
            .header("Authorization", "Basic " + Base64.getEncoder().encodeToString(
                (settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
            .POST(HttpRequest.BodyPublishers.ofString("grant_type=client_credentials&scope=magic-auth"))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            if (response.statusCode() == 401) {
                throw new RuntimeException("[GlideClient] Invalid client credentials");
            } else if (response.statusCode() == 400) {
                Map<String, Object> errorBody = objectMapper.readValue(response.body(), Map.class);
                if ("invalid_scope".equals(errorBody.get("error"))) {
                    throw new RuntimeException("[GlideClient] Client does not have required scopes to access this method");
                }
                throw new RuntimeException("[GlideClient] Invalid request");
            }
            throw new RuntimeException("Failed to generate new session: " + response.body());
        }

        Map<String, Object> body = objectMapper.readValue(response.body(), Map.class);
        return new Session(
            (String) body.get("access_token"),
            System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
            Arrays.asList(((String) body.get("scope")).split(" "))
        );
    }
}
