package com.glideapi.services; 

import com.glideapi.Types.*;
import com.glideapi.Utils;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.*;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;

public class TelcoFinderClient {
    private final GlideSdkSettings settings;
    private Session session = null;
    private static final ObjectMapper objectMapper = new ObjectMapper();

    public TelcoFinderClient(GlideSdkSettings settings) {
        this.settings = settings;
    }


    public static class TelcoFinderNetworkIdResponse {
        public String networkId;
    }

    public static class TelcoFinderSearchResponse {
        public String subject;
        public Properties properties;
        public List<Link> links;

        public static class Properties {
            public String operatorId;
        }

        public static class Link {
            public String rel;
            public String href;
        }
    }

    public TelcoFinderNetworkIdResponse networkIdForNumber(String phoneNumber, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = getSession(conf);
        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/telco-finder/v1/resolve-network-id"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString("{\"phoneNumber\":\"" + Utils.formatPhoneNumber(phoneNumber) + "\"}"))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            if (response.statusCode() == 404) {
                throw new RuntimeException("[GlideClient] Network ID not found for number " + phoneNumber);
            }
            throw new RuntimeException("[GlideClient] Failed to resolve network id: " + response.body());
        }

        return objectMapper.readValue(response.body(), TelcoFinderNetworkIdResponse.class);
    }

    public TelcoFinderSearchResponse lookupIp(String ip, ApiConfig conf) throws Exception {
        return lookup("ipport:" + ip, conf);
    }

    public TelcoFinderSearchResponse lookupNumber(String phoneNumber, ApiConfig conf) throws Exception {
        return lookup("tel:" + Utils.formatPhoneNumber(phoneNumber), conf);
    }

    private TelcoFinderSearchResponse lookup(String subject, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = getSession(conf);
        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/telco-finder/v1/search"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString("{\"resource\":\"" + subject + "\"}"))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            if (response.statusCode() == 404) {
                throw new RuntimeException("[GlideClient] Lookup failed for subject " + subject);
            }
            throw new RuntimeException("[GlideClient] Failed to lookup telco: " + response.body());
        }

        TelcoFinderSearchResponse resObj = objectMapper.readValue(response.body(), TelcoFinderSearchResponse.class);
        resObj.properties.operatorId = resObj.properties.operatorId;
        return resObj;
    }

    private Session getSession(ApiConfig conf) throws Exception {
        if (conf.getSession() != null) {
            return conf.getSession();
        }
        if (session != null && session.getExpiresAt() > System.currentTimeMillis() + (60 * 1000) && 
            session.getScopes().stream().anyMatch(s -> s.endsWith("telco-finder"))) {
            return session;
        }
        return generateNewSession();
    }

    private Session generateNewSession() throws Exception {
        if (settings.getClientId() == null || settings.getClientSecret() == null) {
            throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
            .header("Content-Type", "application/x-www-form-urlencoded")
            .header("Authorization", "Basic " + Base64.getEncoder().encodeToString((settings.getClientId() + ":" + settings.getClientSecret()).getBytes()))
            .POST(HttpRequest.BodyPublishers.ofString("grant_type=client_credentials&scope=telco-finder"))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            throw new RuntimeException("Failed to generate new session: " + response.body());
        }

        Map<String, Object> body = objectMapper.readValue(response.body(), Map.class);
        Session newSession = new Session(
            (String) body.get("access_token"),
            System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
            Arrays.asList(((String) body.get("scope")).split(" "))
        );

        this.session = newSession;
        return newSession;
    }
}
