
package com.glideapi.services;

import com.glideapi.Types.*;
import com.glideapi.Utils;
import com.glideapi.session.SessionStrategy;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.*;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

public class NumberVerifyClient {
    private final GlideSdkSettings settings;
    private static final ObjectMapper objectMapper = new ObjectMapper();

    public NumberVerifyClient(GlideSdkSettings settings) {
        this(settings, false, null);
    }

    public NumberVerifyClient(GlideSdkSettings settings, boolean autoSession, SessionStrategy sessionStrategy) {
        // autoSession and sessionStrategy are not used for NumberVerify
        // as it uses authorization code flow through NumberVerifyUserClient
        this.settings = settings;
    }

    public static class NumberVerifyAuthUrlInput {
        public String state;
        public String useDevNumber;
        public Boolean printCode;

        public NumberVerifyAuthUrlInput(String useDevNumber, Boolean printCode) {
            this.useDevNumber = useDevNumber;
            this.printCode = printCode;
        }
        public NumberVerifyAuthUrlInput(String useDevNumber) {
            this(useDevNumber, false);
        }
        public NumberVerifyAuthUrlInput() {
            this.useDevNumber = null;
            this.printCode = false;
        }
    }

    public static class NumberVerifyClientForParams {
        public String code;
        public String phoneNumber;

        public NumberVerifyClientForParams(String code, String phoneNumber) {
            this.code = code;
            this.phoneNumber = phoneNumber;
        }
        public NumberVerifyClientForParams() {}
    }

    public static class NumberVerifyInput {
        public String phoneNumber;

        public NumberVerifyInput(String phoneNumber) {
            this.phoneNumber = phoneNumber;
        }
        public NumberVerifyInput() {}
    }

    public static class NumberVerifyResponse {
        public boolean devicePhoneNumberVerified;
    }

    public class NumberVerifyUserClient {
        private final String code;
        private Session session = null;
        private final String phoneNumber;

        public NumberVerifyUserClient(NumberVerifyClientForParams params) {
            this.code = params.code;
            this.phoneNumber = params.phoneNumber;
        }

        public void startSession() throws Exception {
            if (settings.getInternal().getAuthBaseUrl() == null) {
                throw new IllegalStateException("[GlideClient] internal.authBaseUrl is unset");
            }

            if (settings.getClientId() == null || settings.getClientSecret() == null) {
                throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
            }

            if (code == null) {
                throw new IllegalStateException("[GlideClient] Code is required to start a session");
            }

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
                .header("Content-Type", "application/x-www-form-urlencoded")
                .header("Authorization", "Basic " + Base64.getEncoder().encodeToString(
                    (settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
                .POST(HttpRequest.BodyPublishers.ofString(
                    "grant_type=authorization_code&code=" + code))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                throw new Exception("Failed to generate new session: " + response.body());
            }

            Map<String, Object> body = objectMapper.readValue(response.body(), 
                new com.fasterxml.jackson.core.type.TypeReference<Map<String, Object>>() {});
            session = new Session(
                (String) body.get("access_token"),
                System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
                Arrays.asList(((String) body.get("scope")).split(" "))
            );
        }

        public String getOperator() {
            if (session == null) {
                throw new IllegalStateException("[GlideClient] Session is required to get operator");
            }
            String token = session.getAccessToken();
            try {
                String[] tokenParts = token.split("\\.");
                String decodedPayload = new String(Base64.getDecoder().decode(tokenParts[1]), StandardCharsets.UTF_8);
                Map<String, Object> tokenData = objectMapper.readValue(decodedPayload, 
                    new com.fasterxml.jackson.core.type.TypeReference<Map<String, Object>>() {});
                Object extObj = tokenData.get("ext");
                if (extObj instanceof Map<?, ?>) {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> ext = (Map<String, Object>) extObj;
                    return (String) ext.get("operator");
                }
                return "unknown";
            } catch (Exception e) {
                e.printStackTrace();
                return "unknown";
            }
        }

        public NumberVerifyResponse verifyNumber(NumberVerifyInput input, ApiConfig conf) throws Exception {
            String operator = "unknown";
            if (conf != null && conf.getSessionIdentifier() != null) {
                operator = this.getOperator();
                this.reportNumberVerifyMetric(conf.getSessionIdentifier(), "Glide numberVerify start function", operator);
            }
            if (session == null) {
                throw new IllegalStateException("[GlideClient] Session is required to verify a number");
            }

            if (settings.getInternal().getApiBaseUrl() == null) {
                throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
            }

            String phoneNumber = input != null && input.phoneNumber != null ? input.phoneNumber : this.phoneNumber;
            if (phoneNumber == null) {
                throw new IllegalArgumentException("[GlideClient] Phone number is required to verify a number");
            }

            Session sessionToUse = conf != null && conf.getSession() != null ? conf.getSession() : this.session;

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/number-verification/verify"))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + sessionToUse.getAccessToken())
                .POST(HttpRequest.BodyPublishers.ofString("{\"phoneNumber\":\"" + phoneNumber + "\"}"))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                throw new Exception("Failed to verify number: " + response.body());
            }

            if (conf != null && conf.getSessionIdentifier() != null) {
                this.reportNumberVerifyMetric(conf.getSessionIdentifier(), "Glide success", operator);
            }

            NumberVerifyResponse result = objectMapper.readValue(response.body(), NumberVerifyResponse.class);
            if (conf != null && conf.getSessionIdentifier() != null) {
                if (result.devicePhoneNumberVerified) {
                    this.reportNumberVerifyMetric(conf.getSessionIdentifier(), "Glide verified", operator);
                } else {
                    this.reportNumberVerifyMetric(conf.getSessionIdentifier(), "Glide unverified", operator);
                }
            }
            return result;
        }

        private void reportNumberVerifyMetric(String sessionId, String metricName, String operator) {
            if (operator == null) {
                operator = "unknown";
            }
            Utils.reportMetric(new Utils.MetricInfo(
                new Date(),
                sessionId,
                metricName,
                "number-verify",
                settings.getClientId(),
                operator
            ));
        }
    }

    public String getAuthUrl(NumberVerifyAuthUrlInput opts) throws Exception {
        if (settings.getInternal().getAuthBaseUrl() == null) {
            throw new IllegalStateException("[GlideClient] internal.authBaseUrl is unset");
        }
        if (settings.getClientId() == null) {
            throw new IllegalStateException("[GlideClient] Client id is required to generate an auth url");
        }

        String state = opts.state != null ? opts.state : UUID.randomUUID().toString();
        String nonce = UUID.randomUUID().toString();

        return settings.getInternal().getAuthBaseUrl() + "/oauth2/auth?" +
               "client_id=" + java.net.URLEncoder.encode(settings.getClientId(), StandardCharsets.UTF_8) +
               "&response_type=code" +
               (settings.getRedirectUri() != null ? "&redirect_uri=" + java.net.URLEncoder.encode(settings.getRedirectUri(), StandardCharsets.UTF_8) : "") +
               "&scope=openid" +
               "&purpose=dpv:FraudPreventionAndDetection:number-verification" +
               "&state=" + java.net.URLEncoder.encode(state, StandardCharsets.UTF_8) +
               "&nonce=" + java.net.URLEncoder.encode(nonce, StandardCharsets.UTF_8) +
               "&max_age=0" +
               (opts.printCode != null && opts.printCode ? "&dev_print=true" : "") +
               (opts.useDevNumber != null ? "&login_hint=" + java.net.URLEncoder.encode("tel:" + opts.useDevNumber, StandardCharsets.UTF_8) : "");
    }

    public NumberVerifyUserClient forUser(NumberVerifyClientForParams params) throws Exception {
        NumberVerifyUserClient client = new NumberVerifyUserClient(params);
        client.startSession();
        return client;
    }
}