package com.glideapi.services;

import com.glideapi.Types.*;
import com.glideapi.Utils;
import com.glideapi.session.SessionStrategy;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.util.*;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Base64;
import java.nio.charset.StandardCharsets;

public class SimSwapClient {
    private final GlideSdkSettings settings;
    private static final ObjectMapper objectMapper = new ObjectMapper();

    public SimSwapClient(GlideSdkSettings settings) {
        this(settings, false, null);
    }

    public SimSwapClient(GlideSdkSettings settings, boolean autoSession, SessionStrategy sessionStrategy) {
        // autoSession and sessionStrategy are not used for SimSwap
        // as it uses backchannel authentication flow
        this.settings = settings;
    }

    public SimSwapUserClient forUser(UserIdentifier identifier) throws Exception {
        SimSwapUserClient client = new SimSwapUserClient(settings, identifier);
        client.startSession();
        return client;
    }

    public class SimSwapUserClient {
        private final GlideSdkSettings settings;
        private final UserIdentifier identifier;
        private Session session = null;
        public boolean requiresConsent = false;
        private String consentUrl = null;
        private String authReqId = null;

        private SimSwapUserClient(GlideSdkSettings settings, UserIdentifier identifier) {
            this.settings = settings;
            this.identifier = identifier;
        }

        public String getConsentUrl() {
            return consentUrl != null ? consentUrl : "";
        }

        public static class SimSwapCheckParams {
            public String phoneNumber;
            public Integer maxAge;
        }

        public static class SimSwapCheckResponse {
            public boolean swapped;
        }

        public static class SimSwapRetrieveDateParams {
            public String phoneNumber;
        }

        public static class SimSwapRetrieveDateResponse {
            public String latestSimChange;
        }

        public SimSwapCheckResponse check(SimSwapCheckParams params, ApiConfig conf) throws Exception {
            if (conf == null)  conf = new ApiConfig();
            if (settings.getInternal().getApiBaseUrl() == null) {
                throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
            }
            String phoneNumber = params.phoneNumber;
            if (phoneNumber == null && identifier instanceof UserIdentifier.PhoneIdentifier) {
                phoneNumber = ((UserIdentifier.PhoneIdentifier) identifier).phoneNumber;
            }
            if (phoneNumber == null) {
                throw new IllegalArgumentException("[GlideClient] phone number not provided");
            }

            Session session = getSession(conf);
            Map<String, Object> body = new HashMap<>();
            body.put("phoneNumber", Utils.formatPhoneNumber(phoneNumber));
            if (params.maxAge != null) {
                body.put("maxAge", params.maxAge);
            }

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/sim-swap/check"))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + session.getAccessToken())
                .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(body)))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() != 200) {
                throw new RuntimeException("[GlideClient] Failed to check sim swap: " + response.body());
            }

            return objectMapper.readValue(response.body(), SimSwapCheckResponse.class);
        }

        public SimSwapRetrieveDateResponse retrieveDate(SimSwapRetrieveDateParams params, ApiConfig conf) throws Exception {
            if (conf == null) conf = new ApiConfig();
            if (settings.getInternal().getApiBaseUrl() == null) {
                throw new IllegalStateException("[GlideClient] internal.apiBaseUrl is unset");
            }
            String phoneNumber = params.phoneNumber;
            if (phoneNumber == null && identifier instanceof UserIdentifier.PhoneIdentifier) {
                phoneNumber = ((UserIdentifier.PhoneIdentifier) identifier).phoneNumber;
            }
            if (phoneNumber == null) {
                throw new IllegalArgumentException("[GlideClient] phone number not provided");
            }

            Session session = getSession(conf);
            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/sim-swap/retrieve-date"))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + session.getAccessToken())
                .POST(HttpRequest.BodyPublishers.ofString("{\"phoneNumber\":\"" + Utils.formatPhoneNumber(phoneNumber) + "\"}"))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() != 200) {
                throw new RuntimeException("[GlideClient] Failed to retrieve sim swap date: " + response.body());
            }

            return objectMapper.readValue(response.body(), SimSwapRetrieveDateResponse.class);
        }

        public void startSession() throws Exception {
            if (settings.getClientId() == null || settings.getClientSecret() == null) {
                throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
            }

            Map<String, String> params = new HashMap<>();
            if (identifier instanceof UserIdentifier.PhoneIdentifier) {
                params.put("login_hint", "tel:" + Utils.formatPhoneNumber(((UserIdentifier.PhoneIdentifier) identifier).phoneNumber));
            } else if (identifier instanceof UserIdentifier.IpIdentifier) {
                params.put("login_hint", "ipport:" + ((UserIdentifier.IpIdentifier) identifier).ipAddress);
            }
            params.put("scope", "sim-swap");

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/backchannel-authentication"))
                .header("Content-Type", "application/x-www-form-urlencoded")
                .header("Authorization", "Basic " + Base64.getEncoder().encodeToString((settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
                .POST(HttpRequest.BodyPublishers.ofString(Utils.urlEncodeMap(params)))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() != 200) {
                throw new RuntimeException("Failed to start session: " + response.body());
            }

            Map<String, Object> body = objectMapper.readValue(response.body(), 
                new com.fasterxml.jackson.core.type.TypeReference<Map<String, Object>>() {});
            if (body.containsKey("consentUrl")) {
                this.requiresConsent = true;
                this.consentUrl = (String) body.get("consentUrl");
            }
            this.authReqId = (String) body.get("auth_req_id");
        }

        private Session getSession(ApiConfig conf) {
            try {
                if (conf.getSession() != null) {
                    return conf.getSession();
                }
                if (session != null && session.getExpiresAt() > System.currentTimeMillis() + (60 * 1000) && 
                    session.getScopes().stream().anyMatch(s -> s.endsWith("sim-swap"))) {
                    return session;
                }
                return generateNewSession();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        public void pollAndWaitForSession() throws Exception {
            while (true) {
                try {
                    Session session = getSession(new ApiConfig());
                    if (session != null) {
                        return;
                    }
                } catch (Exception e) {
                    // Ignore and continue polling
                }
                Thread.sleep(5000);
            }
        }

        private Session generateNewSession() throws Exception {
            if (settings.getClientId() == null || settings.getClientSecret() == null) {
                throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
            }
            if (authReqId == null) {
                startSession();
            }

            if (authReqId == null) {
                throw new IllegalStateException("[GlideClient] Failed to start session");
            }

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
                .header("Content-Type", "application/x-www-form-urlencoded")
                .header("Authorization", "Basic " + Base64.getEncoder().encodeToString((settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
                .POST(HttpRequest.BodyPublishers.ofString("grant_type=urn:openid:params:grant-type:ciba&auth_req_id=" + authReqId))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() != 200) {
                throw new RuntimeException("Failed to generate new session: " + response.body());
            }

            Map<String, Object> body = objectMapper.readValue(response.body(), 
                new com.fasterxml.jackson.core.type.TypeReference<Map<String, Object>>() {});
            Session newSession = new Session(
                (String) body.get("access_token"),
                System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
                Arrays.asList(((String) body.get("scope")).split(" "))
            );

            this.session = newSession;
            this.authReqId = null;
            return newSession;
        }
    }
}