package com.glideapi.services.dto;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import java.util.Map;
import java.util.HashMap;

/**
 * DTOs for MagicAuth service
 */
public class MagicAuthDtos {
    
    /**
     * PLMN (Public Land Mobile Network) identifier
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class PlmnDto {
        @JsonProperty("mcc")
        private String mcc;
        
        @JsonProperty("mnc")
        private String mnc;
        
        public PlmnDto() {}
        
        public PlmnDto(String mcc, String mnc) {
            this.mcc = mcc;
            this.mnc = mnc;
        }
        
        public String getMcc() { return mcc; }
        public void setMcc(String mcc) { this.mcc = mcc; }
        
        public String getMnc() { return mnc; }
        public void setMnc(String mnc) { this.mnc = mnc; }
    }
    
    /**
     * Consent data for Auth V2
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class ConsentData {
        @JsonProperty("consent_text")
        private String consentText;
        
        @JsonProperty("policy_link")
        private String policyLink;
        
        @JsonProperty("policy_text")
        private String policyText;
        
        public ConsentData() {}
        
        public ConsentData(String consentText, String policyLink, String policyText) {
            this.consentText = consentText;
            this.policyLink = policyLink;
            this.policyText = policyText;
        }
        
        public String getConsentText() { return consentText; }
        public void setConsentText(String consentText) { this.consentText = consentText; }
        
        public String getPolicyLink() { return policyLink; }
        public void setPolicyLink(String policyLink) { this.policyLink = policyLink; }
        
        public String getPolicyText() { return policyText; }
        public void setPolicyText(String policyText) { this.policyText = policyText; }
    }
    
    /**
     * Use case for Auth V2
     */
    public enum UseCase {
        @JsonProperty("GetPhoneNumber")
        GET_PHONE_NUMBER("GetPhoneNumber"),
        
        @JsonProperty("VerifyPhoneNumber")
        VERIFY_PHONE_NUMBER("VerifyPhoneNumber");
        
        private final String value;
        
        UseCase(String value) {
            this.value = value;
        }
        
        public String getValue() { return value; }
        
        @Override
        public String toString() { return value; }
    }
    
    /**
     * Auth V2 Prepare request DTO
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AuthV2PrepDto {
        @JsonProperty("plmn")
        private PlmnDto plmn;
        
        @JsonProperty("phone_number")
        private String phoneNumber;
        
        @JsonProperty("fetch_token")
        private Boolean fetchToken;
        
        @JsonProperty("use_case")
        private UseCase useCase;
        
        @JsonProperty("consent_data")
        private ConsentData consentData;
        
        public AuthV2PrepDto() {}
        
        public PlmnDto getPlmn() { return plmn; }
        public void setPlmn(PlmnDto plmn) { this.plmn = plmn; }
        
        public String getPhoneNumber() { return phoneNumber; }
        public void setPhoneNumber(String phoneNumber) { this.phoneNumber = phoneNumber; }
        
        public Boolean getFetchToken() { return fetchToken; }
        public void setFetchToken(Boolean fetchToken) { this.fetchToken = fetchToken; }
        
        public UseCase getUseCase() { return useCase; }
        public void setUseCase(UseCase useCase) { this.useCase = useCase; }
        
        public ConsentData getConsentData() { return consentData; }
        public void setConsentData(ConsentData consentData) { this.consentData = consentData; }
    }
    
    /**
     * Session payload from prepare response
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SessionPayloadRaw {
        @JsonProperty("session_key")
        private String sessionKey;
        
        @JsonProperty("nonce")
        private String nonce;
        
        @JsonProperty("enc_key")
        private String encKey;
        
        public SessionPayloadRaw() {}
        
        public String getSessionKey() { return sessionKey; }
        public void setSessionKey(String sessionKey) { this.sessionKey = sessionKey; }
        
        public String getNonce() { return nonce; }
        public void setNonce(String nonce) { this.nonce = nonce; }
        
        public String getEncKey() { return encKey; }
        public void setEncKey(String encKey) { this.encKey = encKey; }
    }
    
    /**
     * Digital credential response
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class DigitalCredentialResponse {
        @JsonProperty("vp_token")
        private Map<String, Object> vpToken;
        
        public DigitalCredentialResponse() {}
        
        public Map<String, Object> getVpToken() { return vpToken; }
        public void setVpToken(Map<String, Object> vpToken) { this.vpToken = vpToken; }
    }
    
    /**
     * Session metadata for process credential
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SessionMeta {
        @JsonProperty("terminal_id")
        private String terminalId;
        
        @JsonProperty("app")
        private String app;
        
        @JsonProperty("app_name")
        private String appName;
        
        @JsonProperty("terminal_vendor")
        private String terminalVendor;
        
        @JsonProperty("terminal_model")
        private String terminalModel;
        
        @JsonProperty("terminal_sw_version")
        private String terminalSwVersion;
        
        @JsonProperty("vers")
        private String vers;
        
        @JsonProperty("entitlement_version")
        private String entitlementVersion;
        
        @JsonProperty("operation")
        private String operation;
        
        @JsonProperty("operation_targets")
        private String operationTargets;
        
        @JsonProperty("eap_id")
        private String eapId;
        
        public SessionMeta() {}
        
        // Getters and setters
        public String getTerminalId() { return terminalId; }
        public void setTerminalId(String terminalId) { this.terminalId = terminalId; }
        
        public String getApp() { return app; }
        public void setApp(String app) { this.app = app; }
        
        public String getAppName() { return appName; }
        public void setAppName(String appName) { this.appName = appName; }
        
        public String getTerminalVendor() { return terminalVendor; }
        public void setTerminalVendor(String terminalVendor) { this.terminalVendor = terminalVendor; }
        
        public String getTerminalModel() { return terminalModel; }
        public void setTerminalModel(String terminalModel) { this.terminalModel = terminalModel; }
        
        public String getTerminalSwVersion() { return terminalSwVersion; }
        public void setTerminalSwVersion(String terminalSwVersion) { this.terminalSwVersion = terminalSwVersion; }
        
        public String getVers() { return vers; }
        public void setVers(String vers) { this.vers = vers; }
        
        public String getEntitlementVersion() { return entitlementVersion; }
        public void setEntitlementVersion(String entitlementVersion) { this.entitlementVersion = entitlementVersion; }
        
        public String getOperation() { return operation; }
        public void setOperation(String operation) { this.operation = operation; }
        
        public String getOperationTargets() { return operationTargets; }
        public void setOperationTargets(String operationTargets) { this.operationTargets = operationTargets; }
        
        public String getEapId() { return eapId; }
        public void setEapId(String eapId) { this.eapId = eapId; }
    }
    
    /**
     * Options for process credential
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class ProcessCredentialOptions {
        @JsonProperty("session_meta")
        private SessionMeta sessionMeta;
        
        public ProcessCredentialOptions() {}
        
        public SessionMeta getSessionMeta() { return sessionMeta; }
        public void setSessionMeta(SessionMeta sessionMeta) { this.sessionMeta = sessionMeta; }
    }
    
    /**
     * Auth V2 Process Credential request DTO
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AuthV2ProcessCredentialDto {
        private DigitalCredentialResponse credentialResponse;
        private SessionPayloadRaw session;
        private String phoneNumber;
        private ProcessCredentialOptions options;
        
        public AuthV2ProcessCredentialDto() {}
        
        public DigitalCredentialResponse getCredentialResponse() { return credentialResponse; }
        public void setCredentialResponse(DigitalCredentialResponse credentialResponse) { 
            this.credentialResponse = credentialResponse; 
        }
        
        public SessionPayloadRaw getSession() { return session; }
        public void setSession(SessionPayloadRaw session) { this.session = session; }
        
        public String getPhoneNumber() { return phoneNumber; }
        public void setPhoneNumber(String phoneNumber) { this.phoneNumber = phoneNumber; }
        
        public ProcessCredentialOptions getOptions() { return options; }
        public void setOptions(ProcessCredentialOptions options) { this.options = options; }
    }
    
    /**
     * Magic Auth session data
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class MagicAuthSession {
        private String mcc;
        private String mnc;
        private long expiresAt;
        
        public MagicAuthSession() {}
        
        public MagicAuthSession(String mcc, String mnc, long expiresAt) {
            this.mcc = mcc;
            this.mnc = mnc;
            this.expiresAt = expiresAt;
        }
        
        public String getMcc() { return mcc; }
        public void setMcc(String mcc) { this.mcc = mcc; }
        
        public String getMnc() { return mnc; }
        public void setMnc(String mnc) { this.mnc = mnc; }
        
        public long getExpiresAt() { return expiresAt; }
        public void setExpiresAt(long expiresAt) { this.expiresAt = expiresAt; }
        
        public boolean isExpired() {
            return System.currentTimeMillis() > expiresAt;
        }
    }
    
    /**
     * Auth V2 Prepare response (when eligible)
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AuthV2PrepResponse {
        @JsonProperty("protocol")
        private String protocol;
        
        @JsonProperty("data")
        private Map<String, Object> data;
        
        @JsonProperty("session")
        private SessionPayloadRaw session;
        
        public AuthV2PrepResponse() {}
        
        public String getProtocol() { return protocol; }
        public void setProtocol(String protocol) { this.protocol = protocol; }
        
        public Map<String, Object> getData() { return data; }
        public void setData(Map<String, Object> data) { this.data = data; }
        
        public SessionPayloadRaw getSession() { return session; }
        public void setSession(SessionPayloadRaw session) { this.session = session; }
    }
    
    /**
     * Auth V2 Prepare response (when not eligible)
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AuthV2PrepNotEligibleResponse {
        @JsonProperty("eligible")
        private boolean eligible;
        
        @JsonProperty("carrier_name")
        private String carrierName;
        
        @JsonProperty("reason")
        private String reason;
        
        public AuthV2PrepNotEligibleResponse() {}
        
        public boolean isEligible() { return eligible; }
        public void setEligible(boolean eligible) { this.eligible = eligible; }
        
        public String getCarrierName() { return carrierName; }
        public void setCarrierName(String carrierName) { this.carrierName = carrierName; }
        
        public String getReason() { return reason; }
        public void setReason(String reason) { this.reason = reason; }
    }
    
    /**
     * Authenticate response
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class AuthenticateResponse extends HashMap<String, Object> {
        // Extending HashMap allows this class to hold any fields returned by the server
        // while still providing typed getters for known fields
        
        public AuthenticateResponse() {
            super();
        }
        
        public boolean isSuccess() { 
            return Boolean.TRUE.equals(get("success")); 
        }
        
        public void setSuccess(boolean success) { 
            put("success", success); 
        }
        
        public String getPhoneNumber() { 
            // Check both snake_case and camelCase for compatibility
            Object phoneNumber = get("phone_number");
            if (phoneNumber == null) {
                phoneNumber = get("phoneNumber");
            }
            return phoneNumber != null ? phoneNumber.toString() : null;
        }
        
        public void setPhoneNumber(String phoneNumber) { 
            put("phone_number", phoneNumber); 
        }
        
        public Boolean getVerified() { 
            Object verified = get("verified");
            return verified != null ? Boolean.valueOf(verified.toString()) : null;
        }
        
        public void setVerified(Boolean verified) { 
            put("verified", verified); 
        }
        
        public String getError() { 
            Object error = get("error");
            return error != null ? error.toString() : null;
        }
        
        public void setError(String error) { 
            put("error", error); 
        }
        
        public String getAud() { 
            Object aud = get("aud");
            return aud != null ? aud.toString() : null;
        }
        
        public void setAud(String aud) { 
            put("aud", aud); 
        }
    }
} 