package com.glideapi.session;

import com.glideapi.GlideClient;
import com.glideapi.Types.Session;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Session strategy that maintains one session per thread.
 * This provides a good balance between performance and isolation.
 * Each thread gets its own session, avoiding contention while reusing sessions within the same thread.
 * 
 * Best for: Multi-threaded applications where each thread handles multiple requests.
 */
public class ThreadLocalStrategy implements SessionStrategy {
    private final GlideClient client;
    private final ThreadLocal<ConcurrentHashMap<String, Session>> threadLocalSessions = new ThreadLocal<>();
    
    public ThreadLocalStrategy(GlideClient client) {
        this.client = client;
    }
    
    @Override
    public Session getSession() throws Exception {
        return getSession("magic-auth sim-swap number-verify telco-finder");
    }
    
    @Override
    public Session getSession(String scope) throws Exception {
        ConcurrentHashMap<String, Session> sessions = threadLocalSessions.get();
        if (sessions == null) {
            sessions = new ConcurrentHashMap<>();
            threadLocalSessions.set(sessions);
        }
        
        Session session = sessions.get(scope);
        
        // Check if we have a valid session for this scope
        if (session != null && session.getExpiresAt() > System.currentTimeMillis() + 60000) {
            return session;
        }
        
        // Create new session with the requested scope
        session = client.createSession(scope);
        sessions.put(scope, session);
        return session;
    }
    
    @Override
    public CompletableFuture<Session> getSessionAsync() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return getSession();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }
    
    @Override
    public void clearCache() {
        threadLocalSessions.remove();
    }
} 