package com.glideapi; 

import com.glideapi.services.*;
import com.glideapi.session.*;
import com.glideapi.Types.*;
import io.github.cdimascio.dotenv.Dotenv;
import java.util.Base64;
import java.nio.charset.StandardCharsets;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.URI;
import java.util.Map;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.core.type.TypeReference;

public class GlideClient {
    private final GlideSdkSettings settings;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final SessionStrategy sessionStrategy;
    private final boolean autoSession;
    
    public final TelcoFinderClient telcoFinder;
    public final MagicAuthClient magicAuth;
    public final SimSwapClient simSwap;
    public final NumberVerifyClient numberVerify;

    /**
     * Creates a GlideClient with automatic session management using ThreadLocal strategy
     */
    public GlideClient(GlideSdkSettings settings) {
        this(settings, true, SessionManagement.THREAD_LOCAL);
    }

    /**
     * Creates a GlideClient with specified session management
     */
    @SuppressWarnings("this-escape")
    public GlideClient(GlideSdkSettings settings, boolean autoSession, SessionManagement strategy) {
        Dotenv dotenv = Dotenv.configure().ignoreIfMissing().load();
        GlideSdkSettings defaults = new GlideSdkSettings(
            "",  // Default clientId
            "",  // Default clientSecret
            "",  // Default redirectUri
            new GlideSdkSettings.Internal(
                "https://oidc.gateway-x.io",  // Default authBaseUrl
                "https://api.gateway-x.io"    // Default apiBaseUrl
            )
        );

        // First try constructor values, then env vars, then defaults
        this.settings = mergeSettings(dotenv, defaults, settings);

        if (this.settings.getClientId() == null || this.settings.getClientId().isEmpty()) {
            throw new IllegalArgumentException("clientId is required");
        }

        if (this.settings.getClientSecret() == null || this.settings.getClientSecret().isEmpty()) {
            throw new IllegalArgumentException("clientSecret is required");
        }

        if (this.settings.getInternal().getAuthBaseUrl() == null || this.settings.getInternal().getAuthBaseUrl().isEmpty()) {
            throw new IllegalArgumentException("internal.authBaseUrl is unset");
        }

        this.autoSession = autoSession;
        this.sessionStrategy = autoSession ? createStrategy(strategy) : null;

        this.telcoFinder = new TelcoFinderClient(this.settings, this.autoSession, this.sessionStrategy);
        this.magicAuth = new MagicAuthClient(this.settings, this.autoSession, this.sessionStrategy);
        this.simSwap = new SimSwapClient(this.settings, this.autoSession, this.sessionStrategy);
        this.numberVerify = new NumberVerifyClient(this.settings, this.autoSession, this.sessionStrategy);
    }

    public GlideClient(String clientId, String clientSecret) {
        this(new GlideSdkSettings(clientId, clientSecret, "", new GlideSdkSettings.Internal("", "")), true, SessionManagement.THREAD_LOCAL);
    }

    /**
     * Creates a GlideClient with manual session management (backward compatibility)
     */
    public GlideClient(String clientId, String clientSecret, boolean autoSession) {
        this(new GlideSdkSettings(clientId, clientSecret, "", new GlideSdkSettings.Internal("", "")), autoSession, autoSession ? SessionManagement.THREAD_LOCAL : SessionManagement.MANUAL);
    }

    public GlideClient() {
        this(new GlideSdkSettings(), true, SessionManagement.THREAD_LOCAL);
    }
    
    /**
     * Creates a new OAuth session for API authentication with specified scopes.
     * 
     * @param scope Space-separated list of scopes to request
     * @return A new Session with an access token
     * @throws Exception if session creation fails
     */
    public Session createSession(String scope) throws Exception {
        if (settings.getClientId() == null || settings.getClientSecret() == null) {
            throw new IllegalStateException("[GlideClient] Client credentials are required to generate a new session");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
            .header("Content-Type", "application/x-www-form-urlencoded")
            .header("Authorization", "Basic " + Base64.getEncoder().encodeToString(
                (settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
            .POST(HttpRequest.BodyPublishers.ofString("grant_type=client_credentials&scope=" + scope))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            if (response.statusCode() == 401) {
                throw new RuntimeException("[GlideClient] Invalid client credentials");
            } else if (response.statusCode() == 400) {
                Map<String, Object> errorBody = objectMapper.readValue(response.body(), 
                    new TypeReference<Map<String, Object>>() {});
                if ("invalid_scope".equals(errorBody.get("error"))) {
                    throw new RuntimeException("[GlideClient] Client does not have required scopes to access this method");
                }
                throw new RuntimeException("[GlideClient] Invalid request");
            }
            throw new RuntimeException("Failed to generate new session: " + response.body());
        }

        Map<String, Object> body = objectMapper.readValue(response.body(), 
            new TypeReference<Map<String, Object>>() {});
        return new Session(
            (String) body.get("access_token"),
            System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
            Arrays.asList(((String) body.get("scope")).split(" "))
        );
    }
    
    /**
     * Creates a new OAuth session for API authentication.
     * This is a helper method - users can call this directly or implement their own session management.
     * Requests all available scopes.
     * 
     * @return A new Session with an access token
     * @throws Exception if session creation fails
     */
    public Session createSession() throws Exception {
        return createSession("magic-auth sim-swap number-verify telco-finder");
    }

    /**
     * Creates a new session asynchronously
     * 
     * @return CompletableFuture that will complete with a new Session
     */
    public CompletableFuture<Session> createSessionAsync() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return createSession();
            } catch (Exception e) {
                throw new CompletionException(e);
            }
        });
    }

    private GlideSdkSettings mergeSettings(Dotenv dotenv, GlideSdkSettings defaults, GlideSdkSettings userSettings) {
        return new GlideSdkSettings(
            // First try constructor value, then env var, then default
            userSettings.getClientId() != null && !userSettings.getClientId().isEmpty() 
                ? userSettings.getClientId() 
                : dotenv.get("GLIDE_CLIENT_ID", defaults.getClientId()),
            
            userSettings.getClientSecret() != null && !userSettings.getClientSecret().isEmpty()
                ? userSettings.getClientSecret()
                : dotenv.get("GLIDE_CLIENT_SECRET", defaults.getClientSecret()),
            
            userSettings.getRedirectUri() != null && !userSettings.getRedirectUri().isEmpty()
                ? userSettings.getRedirectUri()
                : dotenv.get("GLIDE_REDIRECT_URI", defaults.getRedirectUri()),
            
            new GlideSdkSettings.Internal(
                userSettings.getInternal().getAuthBaseUrl() != null && !userSettings.getInternal().getAuthBaseUrl().isEmpty()
                    ? userSettings.getInternal().getAuthBaseUrl()
                    : dotenv.get("GLIDE_AUTH_BASE_URL", defaults.getInternal().getAuthBaseUrl()),
                
                userSettings.getInternal().getApiBaseUrl() != null && !userSettings.getInternal().getApiBaseUrl().isEmpty()
                    ? userSettings.getInternal().getApiBaseUrl()
                    : dotenv.get("GLIDE_API_BASE_URL", defaults.getInternal().getApiBaseUrl())
            )
        );
    }

    /**
     * Available session management strategies
     */
    public enum SessionManagement {
        /** Creates a new session for each request - highest isolation, highest overhead */
        PER_REQUEST,
        /** One session per thread - good balance for multi-threaded apps */
        THREAD_LOCAL,
        /** Single cached session - lowest overhead, requires thread-safe usage */
        CACHED,
        /** No automatic session management - manual control required */
        MANUAL
    }

    private SessionStrategy createStrategy(SessionManagement strategy) {
        if (strategy == null || strategy == SessionManagement.MANUAL) {
            return null;
        }
        
        switch (strategy) {
            case PER_REQUEST:
                return new PerRequestStrategy(this);
            case THREAD_LOCAL:
                return new ThreadLocalStrategy(this);
            case CACHED:
                return new CachedStrategy(this);
            default:
                throw new IllegalArgumentException("Unknown strategy: " + strategy);
        }
    }
    
    /**
     * Clear any cached sessions
     */
    public void clearSessionCache() {
        if (sessionStrategy != null) {
            sessionStrategy.clearCache();
        }
    }
    
    /**
     * Get the current session strategy
     */
    public SessionStrategy getSessionStrategy() {
        return sessionStrategy;
    }
    
    /**
     * Check if automatic session management is enabled
     */
    public boolean isAutoSession() {
        return autoSession;
    }
}
