package com.glideapi.exceptions;

import java.util.Map;

/**
 * Exception thrown by MagicAuth service operations
 */
public class MagicAuthError extends Exception {
    private final String code;
    private final int status;
    private final String requestId;
    private final String timestamp;
    private final Object details;

    public MagicAuthError(Map<String, Object> errorResponse) {
        super(extractMessage(errorResponse));
        this.code = (String) errorResponse.getOrDefault("code", "UNKNOWN_ERROR");
        this.status = extractStatus(errorResponse);
        this.requestId = (String) errorResponse.get("requestId");
        this.timestamp = (String) errorResponse.get("timestamp");
        this.details = errorResponse.get("details");
    }

    public MagicAuthError(String code, int status, String message) {
        super(message);
        this.code = code;
        this.status = status;
        this.requestId = null;
        this.timestamp = null;
        this.details = null;
    }

    private static String extractMessage(Map<String, Object> errorResponse) {
        Object messageObj = errorResponse.get("message");
        if (messageObj instanceof String) {
            return (String) messageObj;
        }
        
        // Handle error structure from API
        Object error = errorResponse.get("error");
        if (error instanceof Map<?, ?>) {
            @SuppressWarnings("unchecked")
            Map<String, Object> errorMap = (Map<String, Object>) error;
            Object errorMessage = errorMap.get("message");
            if (errorMessage instanceof String) {
                return (String) errorMessage;
            }
        } else if (error instanceof String) {
            return (String) error;
        }
        
        // Try to get a meaningful message from the error response
        if (errorResponse.containsKey("detail")) {
            Object detail = errorResponse.get("detail");
            if (detail instanceof String) {
                return (String) detail;
            }
        }
        
        return "Unknown error occurred";
    }

    private static int extractStatus(Map<String, Object> errorResponse) {
        Object status = errorResponse.get("status");
        if (status instanceof Integer) {
            return (Integer) status;
        } else if (status instanceof String) {
            try {
                return Integer.parseInt((String) status);
            } catch (NumberFormatException e) {
                return 500;
            }
        }
        return 500;
    }

    /**
     * Check if this error matches a specific error code
     */
    public boolean is(MagicAuthErrorCode errorCode) {
        return this.code.equals(errorCode.getCode());
    }

    public String getCode() {
        return code;
    }

    public int getStatus() {
        return status;
    }

    public String getRequestId() {
        return requestId;
    }

    public String getTimestamp() {
        return timestamp;
    }

    public Object getDetails() {
        return details;
    }

    @Override
    public String toString() {
        return String.format("MagicAuthError{code='%s', status=%d, message='%s', requestId='%s', timestamp='%s'}",
                code, status, getMessage(), requestId, timestamp);
    }
} 