package com.glideapi.services;

import com.glideapi.Types.*;
import com.glideapi.Utils;
import com.glideapi.session.SessionStrategy;
import com.glideapi.exceptions.MagicAuthError;
import com.glideapi.exceptions.MagicAuthErrorCode;
import com.glideapi.services.dto.MagicAuthDtos.*;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.core.type.TypeReference;
import java.util.*;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import java.util.UUID;

public class MagicAuthClient {
    private final GlideSdkSettings settings;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final boolean autoSession;
    private final SessionStrategy sessionStrategy;
    private Session session = null;
    private MagicAuthSession magicAuthSession = null;
    
    // Constants matching the Node.js SDK
    private static final String AGGREGATOR_ID = "glide";

    public MagicAuthClient(GlideSdkSettings settings) {
        this(settings, false, null);
    }

    public MagicAuthClient(GlideSdkSettings settings, boolean autoSession, SessionStrategy sessionStrategy) {
        this.settings = settings;
        this.autoSession = autoSession;
        this.sessionStrategy = sessionStrategy;
    }

    public enum FallbackVerificationChannel {
        SMS,
        EMAIL,
        NO_FALLBACK
    }

    public static class BaseMagicAuthStartProps {
        public String phoneNumber;
        public String email;
        public FallbackVerificationChannel fallbackChannel;
        public String redirectUrl;
        public String state;
        public String deviceIpAddress;
        public String otpConfirmationUrl;
        public String rcsConfirmationUrl;

        public BaseMagicAuthStartProps() {}

        public BaseMagicAuthStartProps setPhoneNumber(String phoneNumber) {
            this.phoneNumber = phoneNumber;
            return this;
        }

        public BaseMagicAuthStartProps setEmail(String email) {
            this.email = email;
            return this;
        }

        public BaseMagicAuthStartProps setFallbackChannel(FallbackVerificationChannel fallbackChannel) {
            this.fallbackChannel = fallbackChannel;
            return this;
        }

        public BaseMagicAuthStartProps setRedirectUrl(String redirectUrl) {
            this.redirectUrl = redirectUrl;
            return this;
        }

        public BaseMagicAuthStartProps setState(String state) {
            this.state = state;
            return this;
        }

        public BaseMagicAuthStartProps setDeviceIpAddress(String deviceIpAddress) {
            this.deviceIpAddress = deviceIpAddress;
            return this;
        }

        public BaseMagicAuthStartProps setOtpConfirmationUrl(String otpConfirmationUrl) {
            this.otpConfirmationUrl = otpConfirmationUrl;
            return this;
        }

        public BaseMagicAuthStartProps setRcsConfirmationUrl(String rcsConfirmationUrl) {
            this.rcsConfirmationUrl = rcsConfirmationUrl;
            return this;
        }
    }

    public static class MagicAuthStartCodeResponse {
        @JsonProperty("type")
        public String type; // "EMAIL" or "SMS"
        
        // Getters and setters
        public String getType() { return type; }
        public void setType(String type) { this.type = type; }
    }

    public static class MagicAuthStartMagicResponse {
        @JsonProperty("type")
        public String type; // "MAGIC"
        
        @JsonProperty("authUrl")
        public String authUrl;
        
        // Getters and setters
        public String getType() { return type; }
        public void setType(String type) { this.type = type; }
        
        public String getAuthUrl() { return authUrl; }
        public void setAuthUrl(String authUrl) { this.authUrl = authUrl; }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class MagicAuthStartResponse {
        @JsonProperty("type")
        public String type;
        
        @JsonProperty("authUrl")
        public String authUrl;
        
        @JsonProperty("flatAuthUrl")
        public String flatAuthUrl;
        
        @JsonProperty("operatorId")
        public String operatorId;
        
        // Getters and setters
        public String getType() { return type; }
        public void setType(String type) { this.type = type; }
        
        public String getAuthUrl() { return authUrl; }
        public void setAuthUrl(String authUrl) { this.authUrl = authUrl; }
        
        public String getFlatAuthUrl() { return flatAuthUrl; }
        public void setFlatAuthUrl(String flatAuthUrl) { this.flatAuthUrl = flatAuthUrl; }
        
        public String getOperatorId() { return operatorId; }
        public void setOperatorId(String operatorId) { this.operatorId = operatorId; }
    }

    public static class MagicAuthVerifyProps {
        public String code;
        public String phoneNumber;
        public String email;
        public String token;
        public String deviceIpAddress;
        
        // Getters and setters
        public String getCode() { return code; }
        public void setCode(String code) { this.code = code; }
        
        public String getPhoneNumber() { return phoneNumber; }
        public void setPhoneNumber(String phoneNumber) { this.phoneNumber = phoneNumber; }
        
        public String getEmail() { return email; }
        public void setEmail(String email) { this.email = email; }
        
        public String getToken() { return token; }
        public void setToken(String token) { this.token = token; }
        
        public String getDeviceIpAddress() { return deviceIpAddress; }
        public void setDeviceIpAddress(String deviceIpAddress) { this.deviceIpAddress = deviceIpAddress; }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class MagicAuthCheckResponse {
        @JsonProperty("verified")
        public boolean verified;
        
        // Getters and setters
        public boolean isVerified() { return verified; }
        public void setVerified(boolean verified) { this.verified = verified; }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class MagicAuthStartServerAuthResponse {
        @JsonProperty("sessionId")
        public String sessionId;
        
        @JsonProperty("authUrl")
        public String authUrl;
        
        // Getters and setters
        public String getSessionId() { return sessionId; }
        public void setSessionId(String sessionId) { this.sessionId = sessionId; }
        
        public String getAuthUrl() { return authUrl; }
        public void setAuthUrl(String authUrl) { this.authUrl = authUrl; }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class MagicAuthCheckServerAuthResponse {
        @JsonProperty("status")
        public String status; // "PENDING" or "COMPLETED"
        
        @JsonProperty("verified")
        public boolean verified;
        
        // Getters and setters
        public String getStatus() { return status; }
        public void setStatus(String status) { this.status = status; }
        
        public boolean isVerified() { return verified; }
        public void setVerified(boolean verified) { this.verified = verified; }
    }

    /**
     * Helper method to parse error responses and throw MagicAuthError
     */
    private void handleErrorResponse(HttpResponse<String> response, String operation) throws Exception {
        Map<String, Object> errorBody = new HashMap<>();
        try {
            errorBody = objectMapper.readValue(response.body(), new TypeReference<Map<String, Object>>() {});
        } catch (Exception e) {
            // If we can't parse the error, create a generic error
            errorBody.put("code", MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode());
            errorBody.put("message", "Failed to " + operation + ": " + response.body());
            errorBody.put("status", response.statusCode());
        }
        
        // Ensure status is set
        if (!errorBody.containsKey("status")) {
            errorBody.put("status", response.statusCode());
        }
        
        throw new MagicAuthError(errorBody);
    }

    public MagicAuthStartResponse startAuth(BaseMagicAuthStartProps props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }
        
        if(conf.getSessionIdentifier() != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide start");
        }
        Session session = conf.getSession() != null ? conf.getSession() : getSession();
        Map<String, Object> data = new HashMap<>();
        
        // Add all non-null fields to the request
        if (props.phoneNumber != null) data.put("phoneNumber", props.phoneNumber);
        if (props.email != null) data.put("email", props.email);
        if (props.fallbackChannel != null) data.put("fallbackChannel", props.fallbackChannel.name());
        if (props.redirectUrl != null) data.put("redirectUrl", props.redirectUrl);
        if (props.state != null) data.put("state", props.state);
        if (props.deviceIpAddress != null) data.put("deviceIpAddress", props.deviceIpAddress);

        // Validate that at least one identifier is provided
        if (props.phoneNumber == null && props.email == null) {
            throw new IllegalArgumentException("Either phoneNumber or email must be provided");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/start"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(data)))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            handleErrorResponse(response, "start auth");
        }

        MagicAuthStartResponse result = objectMapper.readValue(response.body(), MagicAuthStartResponse.class);
        if(conf.getSessionIdentifier() != null && result.getOperatorId() != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide verificationStartRes", result.getOperatorId());
        }
        return result;
    }

    public MagicAuthStartServerAuthResponse startServerAuth(BaseMagicAuthStartProps props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = conf.getSession() != null ? conf.getSession() : getSession();
        Map<String, Object> data = new HashMap<>();
        
        // Add all non-null fields to the request
        if (props.phoneNumber != null) data.put("phoneNumber", props.phoneNumber);
        if (props.email != null) data.put("email", props.email);
        if (props.fallbackChannel != null) data.put("fallbackChannel", props.fallbackChannel.name());
        if (props.redirectUrl != null) data.put("redirectUrl", props.redirectUrl);
        if (props.state != null) data.put("state", props.state);
        if (props.otpConfirmationUrl != null) data.put("otpConfirmationUrl", props.otpConfirmationUrl);
        if (props.rcsConfirmationUrl != null) data.put("rcsConfirmationUrl", props.rcsConfirmationUrl);

        // Validate that at least one identifier is provided
        if (props.phoneNumber == null && props.email == null) {
            throw new IllegalArgumentException("Either phoneNumber or email must be provided");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/start-server-auth"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(data)))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            handleErrorResponse(response, "start server auth");
        }

        return objectMapper.readValue(response.body(), MagicAuthStartServerAuthResponse.class);
    }

    public MagicAuthCheckResponse verifyAuth(MagicAuthVerifyProps props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = conf.getSession() != null ? conf.getSession() : getSession();
        Map<String, String> data = new HashMap<>();

        // Add all non-null fields to the request
        if (props.phoneNumber != null) data.put("phoneNumber", props.phoneNumber);
        if (props.email != null) data.put("email", props.email);
        if (props.code != null) data.put("code", props.code);
        if (props.token != null) data.put("token", props.token);
        if (props.deviceIpAddress != null) data.put("deviceIpAddress", props.deviceIpAddress);

        // Validate that we have a valid combination of fields
        boolean hasValidFields = (props.phoneNumber != null && props.token != null) || // Magic auth
                               (props.email != null && props.code != null) ||          // Email verification
                               (props.phoneNumber != null && props.code != null);      // Phone verification

        if (!hasValidFields) {
            throw new IllegalArgumentException("Invalid combination of fields. Must provide either: " +
                "(phoneNumber + token) or (email + code) or (phoneNumber + code)");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/check"))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(data)))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            handleErrorResponse(response, "verify auth");
        }
        if (conf.getSessionIdentifier() != null) {
            this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide success");
        }
        MagicAuthCheckResponse result = objectMapper.readValue(response.body(), MagicAuthCheckResponse.class);
        if (conf.getSessionIdentifier() != null) {
            if (result.isVerified()) {
                this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide verified");
            } else {
                this.reportMagicAuthMetric(conf.getSessionIdentifier(), "Glide unverified");
            }
        }
        return result;
    }

    public MagicAuthCheckServerAuthResponse checkServerAuth(String sessionId, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }

        Session session = conf.getSession() != null ? conf.getSession() : getSession();

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/verification/check-server-auth?sessionId=" + sessionId))
            .header("Content-Type", "application/json")
            .header("Authorization", "Bearer " + session.getAccessToken())
            .GET()
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            handleErrorResponse(response, "check server auth");
        }

        return objectMapper.readValue(response.body(), MagicAuthCheckServerAuthResponse.class);
    }

    /**
     * Prepare authentication for Digital Credentials API
     * 
     * @param props Authentication preparation parameters
     * @param conf API configuration
     * @return Prepare response containing OpenID4VP request data or eligibility info
     * @throws Exception on API errors
     */
    public Object prepare(AuthV2PrepDto props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }

        try {
            // Validate input
            if (props.getUseCase() == null) {
                throw new IllegalArgumentException("use_case is required");
            }
            
            if (props.getPlmn() == null && props.getPhoneNumber() == null) {
                throw new IllegalArgumentException("Either plmn or phone_number must be provided");
            }
            
            if (props.getUseCase() == UseCase.VERIFY_PHONE_NUMBER && props.getPhoneNumber() == null) {
                throw new IllegalArgumentException("phone_number is required for VerifyPhoneNumber use case");
            }

            Session session = conf.getSession() != null ? conf.getSession() : getSession();
            
            // Build request body
            Map<String, Object> requestBody = new HashMap<>();
            if (props.getPhoneNumber() != null) {
                requestBody.put("phone_number", props.getPhoneNumber());
            }
            if (props.getPlmn() != null) {
                requestBody.put("mcc", props.getPlmn().getMcc());
                requestBody.put("mnc", props.getPlmn().getMnc());
            }
            requestBody.put("nonce", UUID.randomUUID().toString());
            requestBody.put("id", AGGREGATOR_ID);
            requestBody.put("use_case", props.getUseCase().getValue());
            
            if (props.getConsentData() != null) {
                Map<String, String> consentMap = new HashMap<>();
                consentMap.put("consent_text", props.getConsentData().getConsentText());
                consentMap.put("policy_link", props.getConsentData().getPolicyLink());
                consentMap.put("policy_text", props.getConsentData().getPolicyText());
                requestBody.put("consent_data", consentMap);
            }

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getApiBaseUrl() + "/magic-auth/v2/auth/prep"))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + session.getAccessToken())
                .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(requestBody)))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            // Parse response to check if it's an error or eligibility response
            Map<String, Object> responseBody = objectMapper.readValue(response.body(), new TypeReference<Map<String, Object>>() {});
            
            // Check if it's an error response
            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                handleErrorResponse(response, "prepare authentication");
            }

            // Check if the response indicates ineligibility
            if (responseBody.containsKey("eligible") && Boolean.FALSE.equals(responseBody.get("eligible"))) {
                // Return not eligible response
                return objectMapper.convertValue(responseBody, AuthV2PrepNotEligibleResponse.class);
            }
            
            // Otherwise, it's a successful prepare response
            // Update magic auth session from response if available
            if (responseBody.containsKey("mcc") && responseBody.containsKey("mnc")) {
                String mcc = (String) responseBody.get("mcc");
                String mnc = (String) responseBody.get("mnc");
                long expiresAt = System.currentTimeMillis() + (5 * 60 * 1000); // 5 minutes
                this.magicAuthSession = new MagicAuthSession(mcc, mnc, expiresAt);
            }
            
            // Convert to AuthV2PrepResponse
            Object ts43Obj = responseBody.get("ts43");
            if (ts43Obj instanceof Map<?, ?>) {
                @SuppressWarnings("unchecked")
                Map<String, Object> ts43 = (Map<String, Object>) ts43Obj;
                AuthV2PrepResponse prepResponse = new AuthV2PrepResponse();
                prepResponse.setProtocol("openid4vp-v1-unsigned");
                
                // Set data from ts43.request
                Object ts43RequestObj = ts43.get("request");
                if (ts43RequestObj instanceof Map<?, ?>) {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> ts43Request = (Map<String, Object>) ts43RequestObj;
                    Map<String, Object> data = new HashMap<>();
                    data.put("nonce", requestBody.get("nonce"));
                    data.put("response_mode", "dc_api");
                    data.put("response_type", "vp_token");
                    
                    // Build dcql_query
                    Map<String, Object> dcqlQuery = new HashMap<>();
                    List<Map<String, Object>> credentials = new ArrayList<>();
                    Map<String, Object> credential = new HashMap<>();
                    credential.put("id", ts43Request.get("id"));
                    credential.put("format", ts43Request.get("format"));
                    
                    Object metaObj = ts43Request.get("meta");
                    if (metaObj instanceof Map<?, ?>) {
                        @SuppressWarnings("unchecked")
                        Map<String, Object> meta = (Map<String, Object>) metaObj;
                        credential.put("meta", meta);
                    }
                    
                    Object claimsObj = ts43Request.get("claims");
                    if (claimsObj instanceof List<?>) {
                        @SuppressWarnings("unchecked")
                        List<String> claims = (List<String>) claimsObj;
                        credential.put("claims", claims);
                    }
                    
                    credentials.add(credential);
                    dcqlQuery.put("credentials", credentials);
                    data.put("dcql_query", dcqlQuery);
                    
                    prepResponse.setData(data);
                }
                
                // Set session from ts43.session
                Object ts43SessionObj = ts43.get("session");
                if (ts43SessionObj instanceof Map<?, ?>) {
                    @SuppressWarnings("unchecked")
                    Map<String, Object> ts43Session = (Map<String, Object>) ts43SessionObj;
                    SessionPayloadRaw sessionPayload = objectMapper.convertValue(ts43Session, SessionPayloadRaw.class);
                    prepResponse.setSession(sessionPayload);
                }
                
                return prepResponse;
            }
            
            // If we can't parse as expected, return the raw response
            return responseBody;
        } catch (MagicAuthError e) {
            // Re-throw MagicAuthError as-is
            throw e;
        } catch (Exception e) {
            // For other exceptions, re-throw as-is
            throw e;
        }
    }

    /**
     * Process credential response from Digital Credentials API
     * 
     * @param props Credential processing parameters
     * @param conf API configuration
     * @return Authentication result with phone number and verification status
     * @throws Exception on API errors
     */
    public AuthenticateResponse processCredential(AuthV2ProcessCredentialDto props, ApiConfig conf) throws Exception {
        if (conf == null) conf = new ApiConfig();
        if (settings.getInternal().getApiBaseUrl() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] internal.apiBaseUrl is unset");
        }

        try {
            // Validate input
            if (props.getCredentialResponse() == null || props.getSession() == null) {
                throw new IllegalArgumentException("credentialResponse and session are required");
            }

            Session session = conf.getSession() != null ? conf.getSession() : getSession();
            
            // Determine the endpoint based on whether we're verifying or getting a phone number
            String endpoint = props.getPhoneNumber() != null ? 
                "/magic-auth/v2/auth/verify-phone-number" : 
                "/magic-auth/v2/auth/get-phone-number";
            
            // Get VP token from credential response
            String vpToken = null;
            if (props.getCredentialResponse().getVpToken() != null) {
                for (Map.Entry<String, Object> entry : props.getCredentialResponse().getVpToken().entrySet()) {
                    Object value = entry.getValue();
                    if (value instanceof String) {
                        vpToken = (String) value;
                    } else if (value instanceof List && !((List<?>) value).isEmpty()) {
                        vpToken = (String) ((List<?>) value).get(0);
                    }
                    break; // Use the first aggregator's token
                }
            }
            
            if (vpToken == null) {
                throw new IllegalArgumentException("No VP token found in credential response");
            }
            
            // Build request body
            Map<String, Object> requestBody = new HashMap<>();
            Map<String, String> sessionMap = new HashMap<>();
            sessionMap.put("session_key", props.getSession().getSessionKey());
            sessionMap.put("nonce", props.getSession().getNonce());
            sessionMap.put("enc_key", props.getSession().getEncKey());
            requestBody.put("session", sessionMap);
            requestBody.put("ts43_dc", vpToken);
            // TODO need to manage the carrier id by the session
            requestBody.put("carrier_id", "310-260");
            
            if (props.getPhoneNumber() != null) {
                requestBody.put("phone_number", props.getPhoneNumber());
            }
            
            if (props.getOptions() != null && props.getOptions().getSessionMeta() != null) {
                Map<String, String> sessionMetaMap = new HashMap<>();
                SessionMeta meta = props.getOptions().getSessionMeta();
                if (meta.getTerminalId() != null) sessionMetaMap.put("terminal_id", meta.getTerminalId());
                if (meta.getApp() != null) sessionMetaMap.put("app", meta.getApp());
                if (meta.getAppName() != null) sessionMetaMap.put("app_name", meta.getAppName());
                if (meta.getTerminalVendor() != null) sessionMetaMap.put("terminal_vendor", meta.getTerminalVendor());
                if (meta.getTerminalModel() != null) sessionMetaMap.put("terminal_model", meta.getTerminalModel());
                if (meta.getTerminalSwVersion() != null) sessionMetaMap.put("terminal_sw_version", meta.getTerminalSwVersion());
                if (meta.getVers() != null) sessionMetaMap.put("vers", meta.getVers());
                if (meta.getEntitlementVersion() != null) sessionMetaMap.put("entitlement_version", meta.getEntitlementVersion());
                if (meta.getOperation() != null) sessionMetaMap.put("operation", meta.getOperation());
                if (meta.getOperationTargets() != null) sessionMetaMap.put("operation_targets", meta.getOperationTargets());
                if (meta.getEapId() != null) sessionMetaMap.put("eap_id", meta.getEapId());
                requestBody.put("session_meta", sessionMetaMap);
            }

            HttpClient client = HttpClient.newHttpClient();
            HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(settings.getInternal().getApiBaseUrl() + endpoint))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + session.getAccessToken())
                .POST(HttpRequest.BodyPublishers.ofString(objectMapper.writeValueAsString(requestBody)))
                .build();

            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                handleErrorResponse(response, "process credential");
            }

            // Parse response
            Map<String, Object> responseBody = objectMapper.readValue(response.body(), new TypeReference<Map<String, Object>>() {});
            
            // Add success: true to match Node.js SDK behavior
            responseBody.put("success", true);
            
            // Convert the entire response to AuthenticateResponse
            // This will map all fields that exist in the response, just like the Node.js SDK
            return objectMapper.convertValue(responseBody, AuthenticateResponse.class);
        } catch (MagicAuthError e) {
            // Re-throw MagicAuthError as-is
            throw e;
        } catch (Exception e) {
            // For other exceptions, re-throw as-is
            throw e;
        }
    }

    /**
     * Get the current Magic Auth session
     * 
     * @return Current session or null if none exists
     */
    public MagicAuthSession getMagicAuthSession() {
        if (magicAuthSession != null && !magicAuthSession.isExpired()) {
            return magicAuthSession;
        }
        return null;
    }

    /**
     * Set the Magic Auth session
     * 
     * @param session Session to set
     */
    public void setMagicAuthSession(MagicAuthSession session) {
        this.magicAuthSession = session;
    }

    /**
     * Clear the Magic Auth session
     */
    public void clearMagicAuthSession() {
        this.magicAuthSession = null;
    }

    private void reportMagicAuthMetric(String sessionId, String metricName) {
        reportMagicAuthMetric(sessionId, metricName, null);
    }

    private void reportMagicAuthMetric(String sessionId, String metricName, String operator) {
        Utils.reportMetric(new Utils.MetricInfo(
            new Date(),
            sessionId,
            metricName,
            "magic-auth",
            settings.getClientId(),
            operator
        ));
    }

    private Session getSession() throws Exception {
        // If automatic session management is enabled, use the strategy with magic-auth scope
        if (autoSession && sessionStrategy != null) {
            return sessionStrategy.getSession("magic-auth");
        }
        
        // Otherwise fall back to manual session management
        if (this.session != null && 
            this.session.getExpiresAt() > System.currentTimeMillis() && 
            this.session.getScopes().contains("magic-auth")) {
            return this.session;
        }
        Session session = generateNewSession();
        this.session = session;
        return session;
    }

    private Session generateNewSession() throws Exception {
        if (settings.getClientId() == null || settings.getClientSecret() == null) {
            throw new MagicAuthError(MagicAuthErrorCode.INTERNAL_SERVER_ERROR.getCode(), 500, "[GlideClient] Client credentials are required to generate a new session");
        }

        HttpClient client = HttpClient.newHttpClient();
        HttpRequest request = HttpRequest.newBuilder()
            .uri(URI.create(settings.getInternal().getAuthBaseUrl() + "/oauth2/token"))
            .header("Content-Type", "application/x-www-form-urlencoded")
            .header("Authorization", "Basic " + Base64.getEncoder().encodeToString(
                (settings.getClientId() + ":" + settings.getClientSecret()).getBytes(StandardCharsets.UTF_8)))
            .POST(HttpRequest.BodyPublishers.ofString("grant_type=client_credentials&scope=magic-auth"))
            .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            if (response.statusCode() == 401) {
                throw new RuntimeException("[GlideClient] Invalid client credentials");
            } else if (response.statusCode() == 400) {
                Map<String, Object> errorBody = objectMapper.readValue(response.body(), 
                    new TypeReference<Map<String, Object>>() {});
                if ("invalid_scope".equals(errorBody.get("error"))) {
                    throw new RuntimeException("[GlideClient] Client does not have required scopes to access this method");
                }
                throw new RuntimeException("[GlideClient] Invalid request");
            }
            throw new RuntimeException("Failed to generate new session: " + response.body());
        }

        Map<String, Object> body = objectMapper.readValue(response.body(), 
            new TypeReference<Map<String, Object>>() {});
        return new Session(
            (String) body.get("access_token"),
            System.currentTimeMillis() + ((Integer) body.get("expires_in") * 1000L),
            Arrays.asList(((String) body.get("scope")).split(" "))
        );
    }
}
