package com.glideapi.session;

import com.glideapi.GlideClient;
import com.glideapi.Types.Session;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Session strategy that maintains a single cached session across all threads.
 * This provides the best performance but requires thread-safe usage.
 * The session is shared globally, minimizing token generation overhead.
 * 
 * Best for: Single-threaded applications or when you can guarantee thread-safe access.
 */
public class CachedStrategy implements SessionStrategy {
    private final GlideClient client;
    private final ConcurrentHashMap<String, Session> sessionCache = new ConcurrentHashMap<>();
    
    public CachedStrategy(GlideClient client) {
        this.client = client;
    }
    
    @Override
    public synchronized Session getSession() throws Exception {
        return getSession("magic-auth sim-swap number-verify telco-finder");
    }
    
    @Override
    public synchronized Session getSession(String scope) throws Exception {
        Session session = sessionCache.get(scope);
        
        // Check if we have a valid session for this scope
        if (session != null && session.getExpiresAt() > System.currentTimeMillis() + 60000) {
            return session;
        }
        
        // Create new session with the requested scope
        session = client.createSession(scope);
        sessionCache.put(scope, session);
        return session;
    }
    
    @Override
    public CompletableFuture<Session> getSessionAsync() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                return getSession();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }
    
    @Override
    public void clearCache() {
        sessionCache.clear();
    }
} 