/*
 * Decompiled with CFR 0.152.
 */
package com.google.gcloud;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.gcloud.ExceptionHandler;
import com.google.gcloud.RetryParams;
import java.io.InterruptedIOException;
import java.nio.channels.ClosedByInterruptException;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class RetryHelper<V> {
    private static final Logger log = Logger.getLogger(RetryHelper.class.getName());
    private final Stopwatch stopwatch;
    private final Callable<V> callable;
    private final RetryParams params;
    private final ExceptionHandler exceptionHandler;
    private int attemptNumber;
    private static final ThreadLocal<Context> context = new ThreadLocal();

    @VisibleForTesting
    static void setContext(Context ctx) {
        if (ctx == null) {
            context.remove();
        } else {
            context.set(ctx);
        }
    }

    static Context getContext() {
        return context.get();
    }

    @VisibleForTesting
    RetryHelper(Callable<V> callable, RetryParams params, ExceptionHandler exceptionHandler, Stopwatch stopwatch) {
        this.callable = (Callable)Preconditions.checkNotNull(callable);
        this.params = (RetryParams)Preconditions.checkNotNull((Object)params);
        this.stopwatch = (Stopwatch)Preconditions.checkNotNull((Object)stopwatch);
        this.exceptionHandler = (ExceptionHandler)Preconditions.checkNotNull((Object)exceptionHandler);
        exceptionHandler.verifyCaller(callable);
    }

    public String toString() {
        MoreObjects.ToStringHelper toStringHelper = MoreObjects.toStringHelper((Object)this);
        toStringHelper.add("params", (Object)this.params);
        toStringHelper.add("stopwatch", (Object)this.stopwatch);
        toStringHelper.add("attemptNumber", this.attemptNumber);
        toStringHelper.add("callable", this.callable);
        toStringHelper.add("exceptionHandler", (Object)this.exceptionHandler);
        return toStringHelper.toString();
    }

    private V doRetry() throws RetryHelperException {
        this.stopwatch.start();
        while (true) {
            Exception exception;
            ++this.attemptNumber;
            try {
                V value = this.callable.call();
                if (this.attemptNumber > 1 && log.isLoggable(Level.FINE)) {
                    log.fine(this + ": attempt #" + this.attemptNumber + " succeeded");
                }
                return value;
            }
            catch (InterruptedIOException | InterruptedException | ClosedByInterruptException e) {
                if (!this.exceptionHandler.shouldRetry(e)) {
                    RetryInterruptedException.propagate();
                }
                exception = e;
            }
            catch (Exception e) {
                if (!this.exceptionHandler.shouldRetry(e)) {
                    throw new NonRetriableException(e);
                }
                exception = e;
            }
            if (this.attemptNumber >= this.params.retryMaxAttempts() || this.attemptNumber >= this.params.retryMinAttempts() && this.stopwatch.elapsed(TimeUnit.MILLISECONDS) >= this.params.totalRetryPeriodMillis()) {
                throw new RetriesExhaustedException(this + ": Too many failures, giving up", exception);
            }
            long sleepDurationMillis = RetryHelper.getSleepDuration(this.params, this.attemptNumber);
            if (log.isLoggable(Level.FINE)) {
                log.fine(this + ": Attempt #" + this.attemptNumber + " failed [" + exception + "], sleeping for " + sleepDurationMillis + " ms");
            }
            try {
                Thread.sleep(sleepDurationMillis);
                continue;
            }
            catch (InterruptedException e) {
                RetryInterruptedException.propagate();
                continue;
            }
            break;
        }
    }

    @VisibleForTesting
    static long getSleepDuration(RetryParams retryParams, int attemptsSoFar) {
        long initialDelay = retryParams.initialRetryDelayMillis();
        double backoffFactor = retryParams.retryDelayBackoffFactor();
        long maxDelay = retryParams.maxRetryDelayMillis();
        long retryDelay = RetryHelper.getExponentialValue(initialDelay, backoffFactor, maxDelay, attemptsSoFar);
        return (long)((StrictMath.random() / 2.0 + 0.75) * (double)retryDelay);
    }

    private static long getExponentialValue(long initialDelay, double backoffFactor, long maxDelay, int attemptsSoFar) {
        return (long)StrictMath.min((double)maxDelay, StrictMath.pow(backoffFactor, StrictMath.max(1, attemptsSoFar) - 1) * (double)initialDelay);
    }

    public static <V> V runWithRetries(Callable<V> callable) throws RetryHelperException {
        return RetryHelper.runWithRetries(callable, RetryParams.defaultInstance(), ExceptionHandler.defaultInstance());
    }

    public static <V> V runWithRetries(Callable<V> callable, RetryParams params, ExceptionHandler exceptionHandler) throws RetryHelperException {
        return RetryHelper.runWithRetries(callable, params, exceptionHandler, Stopwatch.createUnstarted());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    static <V> V runWithRetries(Callable<V> callable, RetryParams params, ExceptionHandler exceptionHandler, Stopwatch stopwatch) throws RetryHelperException {
        RetryHelper<V> retryHelper = new RetryHelper<V>(callable, params, exceptionHandler, stopwatch);
        Context previousContext = RetryHelper.getContext();
        RetryHelper.setContext(new Context(retryHelper));
        try {
            V v = super.doRetry();
            return v;
        }
        finally {
            RetryHelper.setContext(previousContext);
        }
    }

    static class Context {
        private final RetryHelper<?> helper;

        Context(RetryHelper<?> helper) {
            this.helper = helper;
        }

        public RetryParams getRetryParams() {
            return ((RetryHelper)this.helper).params;
        }

        public int getAttemptNumber() {
            return ((RetryHelper)this.helper).attemptNumber;
        }
    }

    public static final class NonRetriableException
    extends RetryHelperException {
        private static final long serialVersionUID = -2331878521983499652L;

        NonRetriableException(Throwable throwable) {
            super(throwable);
        }
    }

    public static final class RetriesExhaustedException
    extends RetryHelperException {
        private static final long serialVersionUID = 780199686075408083L;

        RetriesExhaustedException(String message) {
            super(message);
        }

        RetriesExhaustedException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static final class RetryInterruptedException
    extends RetryHelperException {
        private static final long serialVersionUID = 1678966737697204885L;

        RetryInterruptedException() {
        }

        public static void propagate() throws RetryInterruptedException {
            Thread.currentThread().interrupt();
            throw new RetryInterruptedException();
        }
    }

    public static class RetryHelperException
    extends RuntimeException {
        private static final long serialVersionUID = -2907061015610448235L;

        RetryHelperException() {
        }

        RetryHelperException(String message) {
            super(message);
        }

        RetryHelperException(Throwable cause) {
            super(cause);
        }

        RetryHelperException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

