package com.groupbyinc.api;

import com.groupbyinc.api.builder.BaseRequestBuilder;
import com.groupbyinc.api.config.ConnectionConfiguration;
import com.groupbyinc.common.apache.http.client.config.RequestConfig;
import com.groupbyinc.common.apache.http.client.methods.CloseableHttpResponse;
import com.groupbyinc.common.apache.http.client.methods.HttpPost;
import com.groupbyinc.common.apache.http.client.methods.HttpRequestBase;
import com.groupbyinc.common.apache.http.entity.StringEntity;
import com.groupbyinc.common.apache.http.impl.client.CloseableHttpClient;
import com.groupbyinc.common.apache.http.impl.client.HttpClientBuilder;
import com.groupbyinc.common.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import com.groupbyinc.common.jackson.Mappers;
import com.groupbyinc.common.util.exception.GroupByUtilException;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.logging.Logger;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singletonMap;

public class Client implements AutoCloseable {

  private static final String APPLICATION_JSON = "application/json";
  private static final Logger LOG = Logger.getLogger(Client.class.getName());

  private final ConnectionConfiguration config;
  private final RequestConfig requestConfig;
  private CloseableHttpClient httpClient;

  private String urlOverride;

  public Client() {
    this(null);
  }

  public Client(String url) {
    this(url, true, new ConnectionConfiguration());
  }

  private Client(String url, boolean compressResponse, ConnectionConfiguration config) {
    if (url != null) {
      try {
        new URI(url);
      } catch (URISyntaxException e) {
        throw new IllegalStateException("Invalid url: " + url);
      }
    }
    this.urlOverride = url;

    this.config = config;
    requestConfig =
        RequestConfig.custom().setConnectTimeout(config.getConnectTimeout()).setConnectionRequestTimeout(config.getConnectionRequestTimeout()).setSocketTimeout(config.getSocketTimeout()).build();

    createClient();
  }

  private void createClient() {
    PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager();
    cm.setMaxTotal(config.getMaxConnections());
    cm.setDefaultMaxPerRoute(config.getMaxConnectionsPerRoute());

    httpClient = HttpClientBuilder.create().setConnectionManager(cm).setDefaultRequestConfig(requestConfig).build();
  }

  @Override
  public void close() throws IOException {
    httpClient.close();
  }

  public <T, R> CloseableHttpResponse jsonPost(BaseRequestBuilder<T, R> request) {
    return httpPost(APPLICATION_JSON, request, singletonMap("Accept", APPLICATION_JSON));
  }

  public <T, R> CloseableHttpResponse httpPost(String contentType, BaseRequestBuilder<T, R> request, Map<String, String> additionalHeaders) {
    String url = urlOverride != null ? urlOverride : request.getUrl();
    HttpPost httpRequest = new HttpPost(url);
    additionalHeaders.forEach(httpRequest::addHeader);

    StringEntity entity = new StringEntity(Mappers.writeValueAsString(request.build()), UTF_8);
    httpRequest.setEntity(entity);
    httpRequest.setHeader("Content-Type", contentType + ";charset=" + UTF_8);
    return httpExecute(httpRequest);
  }

  private CloseableHttpResponse httpExecute(HttpRequestBase request) {
    try {
      return httpClient.execute(request);
    } catch (Exception e) {
      LOG.warning(e.getMessage());
      throw new GroupByUtilException(e.getMessage(), e);
    }
  }

  public <T, R> R send(BaseRequestBuilder<T, R> request) throws IOException {
    CloseableHttpResponse webResponse = jsonPost(request);
    InputStream data = webResponse.getEntity().getContent();
    return Mappers.readValue(data, request.getResultType(), false);
  }
}
