package com.fasterxml.jackson;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.dataformat.cbor.CBORFactory;
import com.fasterxml.jackson.dataformat.cbor.CBORGenerator;
import com.fasterxml.jackson.module.afterburner.AfterburnerModule;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Map;

public class Mappers {

  public static final ObjectMapper CBOR = createMapper(Type.CBOR);
  public static final ObjectMapper JSON = createMapper(Type.JSON);
  public static final SerializationFeature[] NO_EXTRA_FEATURES = new SerializationFeature[]{};
  public static final Charset UTF_8 = Charset.forName("UTF-8");

  private enum Type {
    JSON,
    CBOR,

  }

  private Mappers() {
    // hide utility
  }

  private static ObjectMapper createMapper(Type type) {
    JsonFactory factory;
    switch (type) {
      case CBOR:
        factory = new CBORFactory().enable(CBORGenerator.Feature.WRITE_MINIMAL_INTS).enable(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES).enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);
        break;
      case JSON:
      default:
        factory = new JsonFactory().enable(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES).enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);
        break;
    }

    return new ObjectMapper(factory).registerModule(new AfterburnerModule())
        .setSerializationInclusion(Include.NON_NULL)
        .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
        .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
  }

  public static ObjectReader getStrictReader(boolean binary) {
    return getMapper(binary).reader().with(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES).with(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY);
  }

  private static ObjectMapper getMapper(boolean binary, Module... modules) {
    ObjectMapper mapper = binary ? CBOR : JSON;
    if (modules.length > 0) {
      mapper = mapper.copy();
      for (Module m : modules) {
        mapper.registerModule(m);
      }
    }
    return mapper;
  }

  public static String getMimeType(boolean binary) {
    return binary ? "application/cbor" : "application/json";
  }

  public static byte[] writeValueAsBytes(Object object, boolean binary) {
    return writeValueAsBytes(object, binary, false);
  }

  public static byte[] writeValueAsBytes(Object object, boolean binary, boolean pretty) {
    try {
      if (pretty && !binary) {
        return writeValueAsString(object, true).getBytes(UTF_8);
      } else {
        return getWriter(binary, pretty).writeValueAsBytes(object);
      }
    } catch (JsonProcessingException e) {
      throw new IllegalStateException(e);
    }
  }

  public static String writeValueAsString(Object object) {
    return writeValueAsString(object, false);
  }

  public static String writeValueAsString(Object object, boolean pretty) {
    try {
      String s = getWriter(false, pretty).writeValueAsString(object);
      if (pretty) {
        s = s + DefaultIndenter.SYS_LF;
      }
      return s;
    } catch (JsonProcessingException e) {
      throw new IllegalStateException(e);
    }
  }

  private static ObjectWriter getWriter(boolean binary, boolean pretty) {
    return binary ? CBOR.writer() : getPrettyWriter(pretty);
  }

  private static ObjectWriter getPrettyWriter(boolean pretty) {
    return pretty ? JSON.writerWithDefaultPrettyPrinter() : JSON.writer();
  }

  public static <T> T readValue(byte[] data, Class<T> clazz, boolean binary) {
    try {
      return getMapper(binary).readValue(data, clazz);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(byte[] data, Class<T> clazz, Class<?> mixIn, boolean binary) {
    try {
      return getMapper(binary).copy().addMixIn(clazz, mixIn).readerFor(clazz).readValue(data);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(byte[] data, Class<T> clazz, Class<?> mixIn, boolean binary, DeserializationFeature... deserializationFeatures) {
    try {
      return getMapper(binary).copy().addMixIn(clazz, mixIn).readerFor(clazz).withFeatures(deserializationFeatures).readValue(data);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(byte[] data, Class<T> clazz, boolean binary, DeserializationFeature... deserializationFeatures) {
    try {
      return getMapper(binary).copy().readerFor(clazz).withFeatures(deserializationFeatures).readValue(data);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(byte[] data, Class<T> clazz, boolean binary, Module... modules) {
    try {
      return getMapper(binary, modules).readerFor(clazz).readValue(data);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(byte[] data, TypeReference<T> reference, boolean binary) {
    try {
      return getMapper(binary).readValue(data, reference);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(InputStream data, Class<T> clazz, boolean binary) {
    try {
      return getMapper(binary).readValue(data, clazz);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(InputStream data, TypeReference<T> reference, boolean binary) {
    try {
      return getMapper(binary).readValue(data, reference);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static <T> T readValue(InputStream data, Class<T> clazz, boolean binary, Module... modules) {
    try {
      return getMapper(binary, modules).readerFor(clazz).readValue(data);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static String humanReadable(InputStream data, boolean binary) {
    try {
      return JSON.writeValueAsString(getMapper(binary).readValue(data, new TypeReference<Map<String, Object>>() {
      }));
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }

  public static String humanReadable(byte[] data, boolean binary) {
    return humanReadable(data, binary, false);
  }

  public static String humanReadable(byte[] data, boolean binary, boolean pretty) {
    try {
      return getWriter(false, pretty).writeValueAsString(getMapper(binary).readValue(data, new TypeReference<Map<String, Object>>() {
      }));
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }
}
