/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.cloud.sdk.core.util;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;
import com.google.gson.reflect.TypeToken;
import com.ibm.cloud.sdk.core.util.EnvironmentUtils;
import com.ibm.cloud.sdk.core.util.GsonSingleton;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

public final class CredentialUtils {
    private static final Logger LOG = Logger.getLogger(CredentialUtils.class.getName());
    public static final String PLAN_STANDARD = "standard";
    private static final String DEFAULT_CREDENTIAL_FILE_NAME = "ibm-credentials.env";
    private static final String VCAP_SERVICES = "VCAP_SERVICES";

    private CredentialUtils() {
    }

    public static Map<String, String> getServiceProperties(String serviceName) {
        LOG.log(Level.FINE, "Retrieving config properties for service: {0}", serviceName);
        Map<String, String> props = CredentialUtils.getFileCredentialsAsMap(serviceName);
        if (props.isEmpty()) {
            LOG.log(Level.FINE, "Found no relevant properties in credential file");
            props = CredentialUtils.getEnvCredentialsAsMap(serviceName);
        }
        if (props.isEmpty()) {
            LOG.log(Level.FINE, "Found no relevant properties in environment variables");
            props = CredentialUtils.getVcapCredentialsAsMap(serviceName);
        }
        if (props.isEmpty()) {
            LOG.log(Level.FINE, "Found no relevant properties in VCAP credentials");
            props = CredentialUtils.getSystemPropsCredentialsAsMap(serviceName);
        }
        if (props.isEmpty()) {
            LOG.log(Level.FINE, "Found no relevant properties in JVM system properties");
        }
        LOG.log(Level.FINE, "Retrieved {0} properties", props.size());
        return props;
    }

    public static boolean hasBadStartOrEndChar(String credentialValue) {
        return credentialValue != null && (credentialValue.startsWith("{") || credentialValue.startsWith("\"") || credentialValue.endsWith("}") || credentialValue.endsWith("\""));
    }

    public static Map<String, List<VcapService>> getVcapServicesObj() {
        Map result = null;
        String vcapValue = EnvironmentUtils.getenv(VCAP_SERVICES);
        if (StringUtils.isEmpty(vcapValue)) {
            vcapValue = System.getProperty(VCAP_SERVICES);
        }
        if (StringUtils.isNotEmpty(vcapValue)) {
            Gson gson = GsonSingleton.getGson();
            Type typeToken = new TypeToken<Map<String, List<VcapService>>>(){}.getType();
            try {
                result = (Map)gson.fromJson(vcapValue, typeToken);
            }
            catch (Throwable t) {
                LOG.log(Level.WARNING, "Error parsing VCAP_SERVICES", t);
            }
        }
        return result;
    }

    protected static VcapService getVcapServiceEntry(String serviceName) {
        if (serviceName == null || serviceName.isEmpty()) {
            return null;
        }
        Map<String, List<VcapService>> vcapObj = CredentialUtils.getVcapServicesObj();
        if (vcapObj != null) {
            for (List<VcapService> serviceList : vcapObj.values()) {
                for (VcapService service : serviceList) {
                    if (!serviceName.equals(service.name)) continue;
                    return service;
                }
            }
            List<VcapService> services = vcapObj.get(serviceName);
            if (services != null && !services.isEmpty()) {
                return services.get(0);
            }
        }
        return null;
    }

    private static List<File> getFilesToCheck() {
        ArrayList<File> files = new ArrayList<File>();
        String userSpecifiedPath = EnvironmentUtils.getenv("IBM_CREDENTIALS_FILE");
        String userSpecifiedSystemProp = System.getProperty("IBM_CREDENTIALS_FILE");
        String currentWorkingDirectory = System.getProperty("user.dir");
        String unixHomeDirectory = EnvironmentUtils.getenv("HOME");
        String windowsFirstHomeDirectory = EnvironmentUtils.getenv("HOMEDRIVE") + EnvironmentUtils.getenv("HOMEPATH");
        String windowsSecondHomeDirectory = EnvironmentUtils.getenv("USERPROFILE");
        if (StringUtils.isNotEmpty(userSpecifiedPath)) {
            files.add(new File(userSpecifiedPath));
        }
        if (StringUtils.isNotEmpty(userSpecifiedSystemProp)) {
            files.add(new File(userSpecifiedSystemProp));
        }
        if (StringUtils.isNotEmpty(currentWorkingDirectory)) {
            files.add(new File(String.format("%s/%s", currentWorkingDirectory, DEFAULT_CREDENTIAL_FILE_NAME)));
        }
        if (StringUtils.isNotEmpty(unixHomeDirectory)) {
            files.add(new File(String.format("%s/%s", unixHomeDirectory, DEFAULT_CREDENTIAL_FILE_NAME)));
        }
        if (StringUtils.isNotEmpty(windowsFirstHomeDirectory) && !"nullnull".equals(windowsFirstHomeDirectory)) {
            files.add(new File(String.format("%s/%s", windowsFirstHomeDirectory, DEFAULT_CREDENTIAL_FILE_NAME)));
        }
        if (StringUtils.isNotEmpty(windowsSecondHomeDirectory)) {
            files.add(new File(String.format("%s/%s", windowsSecondHomeDirectory, DEFAULT_CREDENTIAL_FILE_NAME)));
        }
        if (LOG.isLoggable(Level.FINE)) {
            if (files.isEmpty()) {
                LOG.log(Level.FINE, "No candidate credential files detected");
            } else {
                String fileNames = files.stream().map(f -> f.getPath()).collect(Collectors.joining(", "));
                LOG.log(Level.FINE, "Candidate credential files detected: {0}", fileNames);
            }
        }
        return files;
    }

    private static List<String> getFirstExistingFileContents(List<File> files) {
        List<String> credentialFileContents = null;
        try {
            for (File file : files) {
                if (!file.isFile()) continue;
                LOG.log(Level.FINE, "Reading credentials from file: {0}", file.getPath());
                credentialFileContents = IOUtils.readLines((InputStream)new FileInputStream(file), StandardCharsets.UTF_8);
                LOG.log(Level.FINE, "Read {0} lines", credentialFileContents.size());
                break;
            }
        }
        catch (IOException e) {
            LOG.log(Level.SEVERE, "Error reading file: ", e);
        }
        return credentialFileContents;
    }

    static Map<String, String> getFileCredentialsAsMap(String serviceName) {
        List<File> files = CredentialUtils.getFilesToCheck();
        List<String> contents = CredentialUtils.getFirstExistingFileContents(files);
        if (contents != null && !contents.isEmpty()) {
            Map<String, String> props = CredentialUtils.parseCredentials(serviceName, contents);
            if (!props.isEmpty()) {
                LOG.log(Level.FINE, "Found {0} properties in credential file", props.size());
            }
            return props;
        }
        return Collections.emptyMap();
    }

    static Map<String, String> getEnvCredentialsAsMap(String serviceName) {
        Map<String, String> env = EnvironmentUtils.getenv();
        if (env != null && !env.isEmpty()) {
            HashMap<String, String> props = new HashMap<String, String>();
            serviceName = serviceName.toUpperCase().replaceAll("-", "_") + "_";
            for (Map.Entry<String, String> entry : env.entrySet()) {
                String credentialName;
                String key = entry.getKey().trim();
                String value = entry.getValue().trim();
                if (!key.startsWith(serviceName) || !StringUtils.isNotEmpty(credentialName = key.substring(serviceName.length())) || !StringUtils.isNotEmpty(value)) continue;
                props.put(credentialName, value);
            }
            if (!props.isEmpty()) {
                LOG.log(Level.FINE, "Found {0} properties in environment variables", props.size());
            }
            return props;
        }
        return Collections.emptyMap();
    }

    static Map<String, String> getSystemPropsCredentialsAsMap(String serviceName) {
        Properties systemProperties = System.getProperties();
        if (systemProperties != null) {
            HashMap<String, String> props = new HashMap<String, String>();
            serviceName = serviceName.toUpperCase().replaceAll("-", "_") + "_";
            for (Map.Entry<Object, Object> entry : systemProperties.entrySet()) {
                String credentialName;
                if (entry.getKey() == null || entry.getValue() == null) continue;
                String key = String.valueOf(entry.getKey()).trim();
                String value = String.valueOf(entry.getValue()).trim();
                if (!StringUtils.isNotEmpty(key) || !key.startsWith(serviceName) || !StringUtils.isNotEmpty(credentialName = key.substring(serviceName.length())) || !StringUtils.isNotEmpty(value)) continue;
                props.put(credentialName, value);
            }
            if (!props.isEmpty()) {
                LOG.log(Level.FINE, "Found {0} properties in JVM system properties", props.size());
            }
            return props;
        }
        return Collections.emptyMap();
    }

    static Map<String, String> getVcapCredentialsAsMap(String serviceName) {
        HashMap<String, String> props = new HashMap<String, String>();
        VcapService vcapService = CredentialUtils.getVcapServiceEntry(serviceName);
        if (vcapService != null && vcapService.credentials != null) {
            CredentialUtils.addToMap(props, "USERNAME", vcapService.credentials.username);
            CredentialUtils.addToMap(props, "PASSWORD", vcapService.credentials.password);
            CredentialUtils.addToMap(props, "URL", vcapService.credentials.url);
            CredentialUtils.addToMap(props, "AUTH_URL", vcapService.credentials.iamUrl);
            CredentialUtils.addToMap(props, "APIKEY", vcapService.credentials.iamApiKey);
            CredentialUtils.addToMap(props, "APIKEY", vcapService.credentials.apikey);
            if (StringUtils.isNotEmpty((CharSequence)props.get("APIKEY"))) {
                CredentialUtils.addToMap(props, "AUTH_TYPE", "iam");
            } else if (StringUtils.isNotEmpty((CharSequence)props.get("USERNAME")) || StringUtils.isNotEmpty((CharSequence)props.get("PASSWORD"))) {
                CredentialUtils.addToMap(props, "AUTH_TYPE", "basic");
            }
        }
        return props;
    }

    private static void addToMap(Map<String, String> map, String key, String value) {
        if (StringUtils.isNotEmpty(value)) {
            value = value.trim();
            map.put(key, value);
        }
    }

    private static Map<String, String> parseCredentials(String serviceName, List<String> contents) {
        HashMap<String, String> props = new HashMap<String, String>();
        serviceName = serviceName.toUpperCase().replaceAll("-", "_") + "_";
        for (String line : contents) {
            String credentialName;
            String[] lineTokens;
            if (line.startsWith("#") || line.trim().isEmpty() || (lineTokens = line.split("=", 2)).length != 2 || lineTokens.length == 0) continue;
            String key = lineTokens[0].trim();
            String value = lineTokens[1].trim();
            if (!key.startsWith(serviceName) || !StringUtils.isNotEmpty(credentialName = key.substring(serviceName.length())) || !StringUtils.isNotEmpty(value)) continue;
            props.put(credentialName, value);
        }
        return props;
    }

    public static class VcapService {
        public String name;
        public String label;
        public String plan;
        public List<String> tags;
        public VcapCredentials credentials;
    }

    public static class VcapCredentials {
        public String url;
        public String username;
        public String password;
        public String apikey;
        @SerializedName(value="iam_apikey")
        public String iamApiKey;
        @SerializedName(value="iam_url")
        public String iamUrl;
        @SerializedName(value="iam_apikey_description")
        public String iamApikeyDescription;
        @SerializedName(value="iam_apikey_name")
        public String iamApikeyName;
        @SerializedName(value="iam_role_crn")
        public String iamRoleCrn;
        @SerializedName(value="iam_serviceid_crn")
        public String iamServiceCrn;
    }
}

