package com.neko233.json.deserializer;

import com.neko233.json.constant.JsonConstant;
import com.neko233.json.utils.StringUtilsForJson;
import com.neko233.json.utils.TextTokenUtils;

import java.util.*;

public class JsonDeserializer {

    private int currentIndex;
    private String jsonString;

    /**
     * 解析 JSON
     *
     * @param json JSON Text
     * @return List<Map < String, Object>>
     */
    public List<Map<String, Object>> parseJson(String json) {
        if (StringUtilsForJson.isBlank(json)) {
            return null;
        }

        currentIndex = 0;
        jsonString = TextTokenUtils.removeCharInTargetRange(json);

        List<Map<String, Object>> resultList = new ArrayList<>();

        if (jsonString.startsWith("{")) {
            // 解析单个 JSON 对象
            Map<String, Object> jsonObject = parseObject();
            resultList.add(jsonObject);
        } else if (jsonString.startsWith("[")) {
            // 解析 JSON 数组
            List<Object> jsonArray = parseArray();
            for (Object item : jsonArray) {
                if (item instanceof Map) {
                    // 已经是 Map 类型，直接添加到结果列表
                    resultList.add((Map<String, Object>) item);
                    continue;
                }
                // 单个数据, 类型判断, json 中只有这几个类型
                if (item instanceof Number
                        || item instanceof String
                        || item instanceof Boolean) {
                    resultList.add(Collections.singletonMap(JsonConstant.DEFAULT_VALUE_KEY, item));
                }
            }
        }

        return resultList;
    }


    private Map<String, Object> parseObject() {
        Map<String, Object> map = new LinkedHashMap<>();

        // Skip the opening curly brace
        currentIndex++;

        while (currentIndex < jsonString.length()) {
            char currentChar = jsonString.charAt(currentIndex);

            if (currentChar == '}') {
                break;
            } else if (currentChar == ',') {
                currentIndex++;
                continue;
            }

            String key = parseString();
            // Skip the colon
            currentIndex++;


            if (StringUtilsForJson.isBlank(key)) {
                continue;
            }
            Object value = parseValue();
            map.put(key, value);
        }

        currentIndex++; // Skip the closing curly brace

        return map;
    }


    /**
     * 解析数组
     *
     * @return 数组
     */
    private List<Object> parseArray() {
        List<Object> list = new ArrayList<>();

        // Skip the opening square bracket
        currentIndex++;

        while (currentIndex < jsonString.length()) {
            char currentChar = jsonString.charAt(currentIndex);

            if (currentChar == ']') {
                break;
            } else if (currentChar == ',') {
                currentIndex++;
                continue;
            }

            Object value = parseValue();
            list.add(value);
        }

        // Skip the closing square bracket
        currentIndex++;

        return list;
    }


    private Object parseValue() {
        char currentChar = jsonString.charAt(currentIndex);

        if (currentChar == '\"') {
            return parseString();
        } else if (currentChar == '{') {
            return parseObject();
        } else if (currentChar == '[') {
            return parseArray();
        } else {
            return parsePrimitive();
        }
    }

    /**
     * 解析字符串, 可以是 key / value
     *
     * @return 文本
     */
    private String parseString() {
        // 跳过开头的 '\"'
        currentIndex++;
        StringBuilder builder = new StringBuilder();

        while (currentIndex < jsonString.length()) {
            char currentChar = jsonString.charAt(currentIndex);

            if (currentChar == '\"') {
                currentIndex++; // 跳过结尾的 '\"'
                break;
            }

            builder.append(currentChar);
            currentIndex++;
        }

        return builder.toString();
    }

    /**
     * 解析原语
     *
     * @return value
     */
    private Object parsePrimitive() {
        StringBuilder valueBuilder = new StringBuilder();

        while (currentIndex < jsonString.length()) {
            char currentChar = jsonString.charAt(currentIndex);

            if (currentChar == ',' || currentChar == ']' || currentChar == '}') {
                break;
            }

            valueBuilder.append(currentChar);
            currentIndex++;
        }

        String value = valueBuilder.toString().trim();

        if (value.isEmpty()) {
            return null;
        }
        if (value.equalsIgnoreCase("null")) {
            return null;
        }
        if (value.equalsIgnoreCase("true")) {
            return true;
        }
        if (value.equalsIgnoreCase("false")) {
            return false;
        }
        // 小数
        if (value.contains(".") || value.contains("e") || value.contains("E")) {
            // 小数, 优先
            return Double.parseDouble(value);
        }

        // 长整数
        return Long.parseLong(value);
    }

}
