/*
 * Decompiled with CFR 0.152.
 */
package org.dataloader;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import org.dataloader.BatchLoader;
import org.dataloader.CacheMap;
import org.dataloader.DataLoaderOptions;
import org.dataloader.Try;
import org.dataloader.impl.Assertions;
import org.dataloader.impl.CompletableFutureKit;
import org.dataloader.stats.Statistics;
import org.dataloader.stats.StatisticsCollector;

public class DataLoader<K, V> {
    private final BatchLoader<K, V> batchLoadFunction;
    private final DataLoaderOptions loaderOptions;
    private final CacheMap<Object, CompletableFuture<V>> futureCache;
    private final List<AbstractMap.SimpleImmutableEntry<K, CompletableFuture<V>>> loaderQueue;
    private final StatisticsCollector stats;

    public static <K, V> DataLoader<K, V> newDataLoader(BatchLoader<K, V> batchLoadFunction) {
        return DataLoader.newDataLoader(batchLoadFunction, null);
    }

    public static <K, V> DataLoader<K, V> newDataLoader(BatchLoader<K, V> batchLoadFunction, DataLoaderOptions options) {
        return new DataLoader<K, V>(batchLoadFunction, options);
    }

    public static <K, V> DataLoader<K, V> newDataLoaderWithTry(BatchLoader<K, Try<V>> batchLoadFunction) {
        return DataLoader.newDataLoaderWithTry(batchLoadFunction, null);
    }

    public static <K, V> DataLoader<K, V> newDataLoaderWithTry(BatchLoader<K, Try<V>> batchLoadFunction, DataLoaderOptions options) {
        return new DataLoader<K, Try<V>>(batchLoadFunction, options);
    }

    public DataLoader(BatchLoader<K, V> batchLoadFunction) {
        this(batchLoadFunction, null);
    }

    public DataLoader(BatchLoader<K, V> batchLoadFunction, DataLoaderOptions options) {
        this.batchLoadFunction = Assertions.nonNull(batchLoadFunction);
        this.loaderOptions = options == null ? new DataLoaderOptions() : options;
        this.futureCache = this.determineCacheMap(this.loaderOptions);
        this.loaderQueue = new ArrayList<AbstractMap.SimpleImmutableEntry<K, CompletableFuture<V>>>();
        this.stats = Assertions.nonNull(this.loaderOptions.getStatisticsCollector());
    }

    private CacheMap<Object, CompletableFuture<V>> determineCacheMap(DataLoaderOptions loaderOptions) {
        return loaderOptions.cacheMap().isPresent() ? loaderOptions.cacheMap().get() : CacheMap.simpleMap();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<V> load(K key) {
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            Object cacheKey = this.getCacheKey(Assertions.nonNull(key));
            this.stats.incrementLoadCount();
            boolean batchingEnabled = this.loaderOptions.batchingEnabled();
            boolean cachingEnabled = this.loaderOptions.cachingEnabled();
            if (cachingEnabled && this.futureCache.containsKey(cacheKey)) {
                this.stats.incrementCacheHitCount();
                return this.futureCache.get(cacheKey);
            }
            CompletionStage future = new CompletableFuture();
            if (batchingEnabled) {
                this.loaderQueue.add(new AbstractMap.SimpleImmutableEntry(key, future));
            } else {
                this.stats.incrementBatchLoadCountBy(1L);
                CompletableFuture<List<V>> batchedLoad = this.batchLoadFunction.load(Collections.singletonList(key)).toCompletableFuture();
                future = batchedLoad.thenApply(list -> list.get(0));
            }
            if (cachingEnabled) {
                this.futureCache.set(cacheKey, future);
            }
            return future;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<List<V>> loadMany(List<K> keys) {
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            List collect = keys.stream().map(this::load).collect(Collectors.toList());
            return CompletableFutureKit.allOf(collect);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<List<V>> dispatch() {
        boolean batchingEnabled = this.loaderOptions.batchingEnabled();
        ArrayList keys = new ArrayList();
        ArrayList queuedFutures = new ArrayList();
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            this.loaderQueue.forEach(entry -> {
                keys.add(entry.getKey());
                queuedFutures.add((CompletableFuture)entry.getValue());
            });
            this.loaderQueue.clear();
        }
        if (!batchingEnabled || keys.size() == 0) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }
        int maxBatchSize = this.loaderOptions.maxBatchSize();
        if (maxBatchSize > 0 && maxBatchSize < keys.size()) {
            return this.sliceIntoBatchesOfBatches(keys, queuedFutures, maxBatchSize);
        }
        return this.dispatchQueueBatch(keys, queuedFutures);
    }

    private CompletableFuture<List<V>> sliceIntoBatchesOfBatches(List<K> keys, List<CompletableFuture<V>> queuedFutures, int maxBatchSize) {
        ArrayList allBatches = new ArrayList();
        int len = keys.size();
        int batchCount = (int)Math.ceil((double)len / (double)maxBatchSize);
        for (int i = 0; i < batchCount; ++i) {
            int fromIndex = i * maxBatchSize;
            int toIndex = Math.min((i + 1) * maxBatchSize, len);
            List<K> subKeys = keys.subList(fromIndex, toIndex);
            List<CompletableFuture<V>> subFutures = queuedFutures.subList(fromIndex, toIndex);
            allBatches.add(this.dispatchQueueBatch(subKeys, subFutures));
        }
        return CompletableFuture.allOf(allBatches.toArray(new CompletableFuture[allBatches.size()])).thenApply(v -> allBatches.stream().map(CompletableFuture::join).flatMap(Collection::stream).collect(Collectors.toList()));
    }

    private CompletableFuture<List<V>> dispatchQueueBatch(List<K> keys, List<CompletableFuture<V>> queuedFutures) {
        CompletionStage<Object> batchLoad;
        this.stats.incrementBatchLoadCountBy(keys.size());
        try {
            batchLoad = Assertions.nonNull(this.batchLoadFunction.load(keys), "Your batch loader function MUST return a non null CompletionStage promise");
        }
        catch (Exception e) {
            batchLoad = CompletableFutureKit.failedFuture(e);
        }
        return ((CompletableFuture)batchLoad.toCompletableFuture().thenApply(values2 -> {
            Assertions.assertState(keys.size() == values2.size(), "The size of the promised values MUST be the same size as the key list");
            for (int idx = 0; idx < queuedFutures.size(); ++idx) {
                Object value = values2.get(idx);
                CompletableFuture future = (CompletableFuture)queuedFutures.get(idx);
                if (value instanceof Throwable) {
                    this.stats.incrementLoadErrorCount();
                    future.completeExceptionally((Throwable)value);
                    continue;
                }
                if (value instanceof Try) {
                    Try tryValue = (Try)value;
                    if (tryValue.isSuccess()) {
                        future.complete(tryValue.get());
                        continue;
                    }
                    this.stats.incrementLoadErrorCount();
                    future.completeExceptionally(tryValue.getThrowable());
                    continue;
                }
                Object val = value;
                future.complete(val);
            }
            return values2;
        })).exceptionally(ex -> {
            this.stats.incrementBatchLoadExceptionCount();
            for (int idx = 0; idx < queuedFutures.size(); ++idx) {
                Object key = keys.get(idx);
                CompletableFuture future = (CompletableFuture)queuedFutures.get(idx);
                future.completeExceptionally((Throwable)ex);
                this.clear(key);
            }
            return Collections.emptyList();
        });
    }

    public List<V> dispatchAndJoin() {
        ArrayList<V> results = new ArrayList<V>();
        List<V> joinedResults = this.dispatch().join();
        results.addAll(joinedResults);
        while (this.dispatchDepth() > 0) {
            joinedResults = this.dispatch().join();
            results.addAll(joinedResults);
        }
        return results;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int dispatchDepth() {
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            return this.loaderQueue.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataLoader<K, V> clear(K key) {
        Object cacheKey = this.getCacheKey(key);
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            this.futureCache.delete(cacheKey);
        }
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataLoader<K, V> clearAll() {
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            this.futureCache.clear();
        }
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DataLoader<K, V> prime(K key, V value) {
        Object cacheKey = this.getCacheKey(key);
        DataLoader dataLoader = this;
        synchronized (dataLoader) {
            if (!this.futureCache.containsKey(cacheKey)) {
                this.futureCache.set(cacheKey, CompletableFuture.completedFuture(value));
            }
        }
        return this;
    }

    public DataLoader<K, V> prime(K key, Exception error) {
        Object cacheKey = this.getCacheKey(key);
        if (!this.futureCache.containsKey(cacheKey)) {
            this.futureCache.set(cacheKey, CompletableFutureKit.failedFuture(error));
        }
        return this;
    }

    public Object getCacheKey(K key) {
        return this.loaderOptions.cacheKeyFunction().isPresent() ? this.loaderOptions.cacheKeyFunction().get().getKey(key) : key;
    }

    public Statistics getStatistics() {
        return this.stats.getStatistics();
    }
}

